package cz.cuni.amis.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.Map.Entry;

import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.exception.PogamutIOException;

public class IniFile {
	
	/**
	 * Class representing one section of the ini file.
	 * @author Jimmy
	 */
	public static class Section {
		
		private String name;
		private Map<String, String> props = new TreeMap<String, String>(new Comparator<String>() {
			@Override
			public int compare(String o1, String o2) {
				if (o1 == null) return -1;
				if (o2 == null) return 1;
				return o1.toLowerCase().compareTo(o2.toLowerCase());
			}
		});
		
		/**
		 * Creates a section of the given name.
		 * <p><p>
		 * Name can't be null!
		 * 
		 * @param name
		 */
		public Section(String name) {
			this.name = name;
			NullCheck.check(this.name, "name");
		}

		/**
		 * Copy-constructor.
		 * @param section
		 */
		public Section(Section section) {
			this.name = section.getName();
			this.add(section);
		}

		/**
		 * Returns name of the section.
		 * @return
		 */
		public String getName() {
			return name;
		}
		
		/**
		 * Sets a property key=value into the section.
		 * @param key
		 * @param value
		 * @return this
		 */
		public Section put(String key, String value) {
			NullCheck.check(key, "key");
			props.put(key, value);
			return this;
		}
		
		/**
		 * Returns a value of the propety with 'key'.
		 * @param key
		 * @return
		 */
		public String get(String key) {
			return props.get(key);
		}
		
		/**
		 * Whether the section contains property of the given key.
		 * @param key
		 * @return
		 */
		public boolean containsKey(String key) {
			return props.containsKey(key);
		}
		
		/**
		 * Returns all keys stored within the map.
		 * @return
		 */
		public Set<String> getKeys() {
			return props.keySet();
		}
		
		/**
		 * Alias for {@link Section#getKeys()}.
		 * @return
		 */
		public Set<String> keySet() {
			return getKeys();
		}
		
		/**
		 * Removes a property under the 'key' from this section.
		 * @param key
		 * @return this
		 */
		public Section remove(String key) {
			props.remove(key);
			return this;
		}
		
		/**
		 * Deletes all properties within this section.
		 * @return
		 */
		public Section clear() {
			props.clear();
			return this;
		}

		/**
		 * Alias for {@link Section#put(String, String)}.
		 * @param key
		 * @param value
		 * @return this
		 */
		public Section set(String key, String value) {		
			return put(key, value);
		}

		/**
		 * Adds all properties from 'section' into this one.
		 * @param section
		 * @return 
		 * @return this
		 */
		public Section add(Section section) {
			for (Entry<String, String> entry : section.props.entrySet()) {
				props.put(entry.getKey(), entry.getValue());
			}
			return this;
		}

		/**
		 * Writes this section into the writer.
		 * @param writer
		 */
		public void output(PrintWriter writer) {
			writer.print("[");
			writer.print(name);
			writer.println("]");
			for (Entry<String, String> entry : props.entrySet()) {
				writer.print(entry.getKey());
				writer.print("=");
				writer.println(entry.getValue());
			}
		}
		
		@Override
		public String toString() {
			return "IniFile.Section[name=" + name + ", entries=" + props.size() + "]";
		}
		
	}
	
	private Map<String, Section> sections = new TreeMap<String, Section>(new Comparator<String>() {
		@Override
		public int compare(String o1, String o2) {
			if (o1 == null) return -1;
			if (o2 == null) return 1;
			return o1.toLowerCase().compareTo(o2.toLowerCase());
		}
	});
	
	/**
	 * Constructs Ini file with no defaults.
	 */
	public IniFile() {
	}
	
	/**
	 * Constructs GameBots2004Ini with defaults taken 'source' (file must exists!).
	 * 
	 * @param source
	 */
	public IniFile(File source) {
		if (!source.exists()) {
			throw new PogamutException("File with defaults does not exist at: " + source.getAbsolutePath() + ".", this);
		}
		load(source);
	}

	public IniFile(IniFile ini) {
		for (Section section : ini.getSections()) {
			addSection(new Section(section));
		}
	}

	/**
	 * Loads {@link IniFile#source} into {@link IniFile#sections}.
	 * <p><p>
	 * Note that his method won't clear anything, it will just load all sections/properties from the given file possibly overwriting existing properties
	 * in existing sections.
	 * 
	 * @param source 
	 */
	public void load(File source) {
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new FileReader(source));
		} catch (FileNotFoundException e) {
			throw new PogamutException("Could not open file with defaults for GameBots2004.ini, file was not found at: " + source.getAbsolutePath() + ", caused by: " + e.getMessage(), e);
		}
		
		Section currSection = null;
		
		try {
			while (reader.ready()) {
				String line = reader.readLine().trim();
				if (line.length() == 0) continue;
				if (line.trim().startsWith(";")) continue;
				if (line.startsWith("[") && line.endsWith("]")) {
					if (currSection != null && getSection(currSection.getName()) == null) {
						addSection(currSection);
					}
					String sectionName = line.substring(1, line.length()-1);
					currSection = getSection(sectionName);
					if (currSection == null) currSection = new Section(sectionName);
				} else {					
					int separ = line.indexOf("=");
					if (separ < 0) {
						// TODO: [Jimmy] throw an exception? at least log somewhere?
						continue;
					}
					if (currSection == null) {
						throw new PogamutException("There is an entry '" + line + "' inside ini file that does not belong to any section. (Source: " + source.getAbsolutePath() + ")", this);
					}
					String key = line.substring(0, separ);					
					String value =
						(separ+1 < line.length() ? 
								line.substring(separ+1, line.length())
							:	"");
					currSection.put(key, value);
				}				
			}
		} catch (IOException e) {
			throw new PogamutIOException("Could not completely read file with defaults for GameBots2004.ini from: " + source.getAbsolutePath() + ", caused by: " + e.getMessage(), e, this);
		} finally {
			try {
				reader.close();
			} catch (IOException e) {
			}
		}
		
		if (currSection != null && getSection(currSection.getName()) == null) {
			addSection(currSection);
		}
	}

	/**
	 * Add all sections from one ini file into this one.
	 * @param iniFile
	 * @return this
	 */
	public IniFile addIniFile(IniFile iniFile) {
		for (Section section : iniFile.sections.values()) {
			addSection(section);
		}
		return this;
	}
	
	/**
	 * Adds a new section into this class (won't overwrite existing one).
	 * 
	 * @param sectionName
	 * @return section that is stored in this class
	 */
	public Section addSection(String sectionName) {
		return addSection(new Section(sectionName));
	}
	
	/**
	 * Adds section into this ini file. If section of the same name exists, it won't be replaced. Instead all properties
	 * from 'section' will be put there.
	 * <p><p>
	 * If 'section' is a new section (i.e., its {@link Section#getName()} is not already present in stored sections), this instance
	 * will be stored here (does not hard-copy the section!). For hard-copy variant, use {@link IniFile#copySection(Section)}.
	 * 
	 * @param section
	 * @return section that is stored in this class
	 */
	public Section addSection(Section section) {
		Section oldSection = sections.get(section.getName());
		if (oldSection != null) {
			oldSection.add(section);
			return oldSection;
		} else {
			sections.put(section.getName(), section);
			return section;
		}
	}
	
	public Section copySection(Section section) {
		Section oldSection = sections.get(section.getName());
		if (oldSection != null) {
			oldSection.add(new Section(section));
			return oldSection;
		} else {
			sections.put(section.getName(), section);
			return section;
		}
	}
	
	public boolean hasSection(String name) {
		return sections.containsKey(name);
	}
	
	public Section getSection(String name) {
		return sections.get(name);
	}
	
	public Set<String> getSectionNames() {
		return sections.keySet();
	}
	
	public Collection<Section> getSections() {
		return sections.values();
	}
	
	public String get(String section, String key) {
		Section sec = sections.get(section);
		if (sec == null) return null;
		return sec.get(key);
	}
	
	/**
	 * Sets property key=value into section 'section'
	 * @param section
	 * @param key
	 * @param value
	 * @return section instance that the property was set into
	 */
	public Section set(String section, String key, String value) {
		Section sec = sections.get(section);
		if (sec == null) {
			sec = addSection(section);
		}
		sec.set(key, value);
		return sec;
	}

	/**
	 * Outputs GameBots2004.ini stored by this class into 'file'. If 'file' exists, it overwrites it.
	 * 
	 * @param file
	 */
	public void output(String pathToFileToBeCreated) {
		NullCheck.check(pathToFileToBeCreated, "pathToFileToBeCreated");
		output(new File(pathToFileToBeCreated));
	}
	
	/**
	 * Outputs GameBots2004.ini stored by this class into 'file'. If 'file' exists, it overwrites it.
	 * 
	 * @param file
	 */
	public void output(File file) {
		NullCheck.check(file, "file");
		PrintWriter writer = null;
		try {
			writer = new PrintWriter(new FileWriter(file));
			output(writer);
		} catch (IOException e) {
			throw new PogamutIOException("Could not write ini file into '" + file.getAbsolutePath() + "', caused by: " + e.getMessage(), e, this);
		} finally {
			writer.close();
		}
	}
	
	/**
	 * Outputs contents of this {@link IniFile} into the 'writer'.
	 * 
	 * @param writer
	 */
	public void output(PrintWriter writer) {
		NullCheck.check(writer, "writer");
		boolean first = true;
		for (Section section : sections.values()) {
			if (first) first = false;
			else writer.println();
			section.output(writer);
		}
	}
	
	/**
	 * Returns contents of this {@link IniFile} as string.
	 * 
	 * @return
	 */
	public String output() {
		StringWriter stringWriter = new StringWriter();
		output(new PrintWriter(stringWriter));
		return stringWriter.toString();
	}

}
