package cz.cuni.amis.pogamut.episodic.episodes;

import java.io.Serializable;

/**
 * The <code>AgeInterval</code> class contains information about the age of
 * a chronobag. Age of all episodes in chronobag should be at least
 * <code>minAge</code> and at most <code>maxAge</code> days.
 * <p>
 * An instance of this class exist in each Chronobag instance.
 *
 * @author  Michal Cermak
 */
public class AgeInterval implements Serializable {
    /**
     * Determines if a de-serialized file is compatible with this class.
     *
     * Maintainers must change this value if and only if the new version
     * of this class is not compatible with old versions. See Sun docs
     * for <a href=http://java.sun.com/products/jdk/1.1/docs/guide
     * /serialization/spec/version.doc.html> details. </a>
     *
     * Not necessary to include in first version of the class, but
     * included here as a reminder of its importance.
     */
    private static final long serialVersionUID = 1L;

    /** Lower bound for age of episodes in a chronobag (in days).*/
    private int minAge;
    /** Upper bound for age of episodes in a chronobag (in days).*/
    private int maxAge;

    /** Instantiate the class by providing the ends of age interval.
     * <p>
     * Both bounds should be zero in case of the chronobag representing
     * the current day. (Episodes are zero days old).
     *
     * @param   min Minimum age of episodes.
     * @param   max Maximum age of episodes.
     */
    public AgeInterval(int min, int max) {
        if (min > max) throw new RuntimeException("Abnormal AgeInterval boundaries!");
        this.minAge  = min;
        this.maxAge = max;
    }

    /** Getter method for minAge variable.
     *
     * @return  Lower bound for age of episodes in a chronobag (in days).
     */
    public int getMinAge()  { return minAge;  }

    /** Getter method for maxAge variable.
     *
     * @return  Upper bound for age of episodes in a chronobag (in days).
     */
    public int getMaxAge() { return maxAge; }

    /** Checks if there is non-zero intersection between the current
     * and the specified interval.
     *
     * @param   b   An interval to check the intersection with.
     * @return  True if the intervals have non-empty intersection,
     * false otherwise.
     */
    public boolean intersects(AgeInterval b) {
        AgeInterval a = this;
        if (b.minAge <= a.maxAge && b.minAge >= a.minAge) { return true; }
        if (a.minAge <= b.maxAge && a.minAge >= b.minAge) { return true; }
        return false;
    }

    /** Joins the interval with another <code>Age Interval</code>.
     * If there is no intersection between the two intervals,
     * their union will not be an interval.
     *
     * @param   other   Interval to join with.
     * @return  If union of both intervals is an interval returns the new
     * unified interval, otherwise returns <code>null</code>.
     */
    public AgeInterval join(AgeInterval other) {
        if (!this.intersects(other)) return null;
        return new AgeInterval(Math.min(minAge, other.minAge), Math.max(maxAge, other.maxAge));
    }

    /** Increases the values of <code>minAge</code> and <code>maxAge</code> by one.
     * <p>
     * In other words, increases the age of interval by one day.
     */
    public void increase() {
        minAge++;
        maxAge++;
    }

    /** Returns a <code>String</code> object representing this
     * <code>AgeInterval</code>'s value. The lower and upper bounds
     * both appears in the representation.
     *
     * @return  a string representation of the value of this object.
     */
    @Override
    public String toString() {
        return "[" + minAge + ", " + maxAge + "]";
    }
}
