package cz.cuni.amis.pogamut.defcon.example;

import java.util.List;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.defcon.ai.AbstractAI;
import cz.cuni.amis.pogamut.defcon.ai.fleetai.IFleetAI;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.SetActionTarget;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.SetMovementTarget;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.SetState;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.WhiteboardDraw;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Carrier;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.City;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConUnitObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConViewableObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Fleet;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Sub;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.BasicFlag;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;
import cz.cuni.amis.pogamut.defcon.consts.state.CarrierState;
import cz.cuni.amis.pogamut.defcon.consts.state.SubState;

/**
 * MixedFleetAI
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class MixedFleetAI extends AbstractAI implements IFleetAI {
	protected final Fleet fleet;
	protected DefConLocation targetLocation;
	protected DefConLocation optimizedTargetLocation;
	protected DefConLocation currentTargetLocation;
	protected DefConLocation originalLocation;
	protected final List<Integer> enemyCityIds;
	protected boolean canLaunch = false;
	protected float arrivalTime = -1;

	protected State currentState = State.PRE_DEFCON4;

	protected enum State {
		PRE_DEFCON4,
		GOTO_ATTACK_POSITION,
		WAIT_FOR_DEFCON1,
		LAUNCH_NUKES,
		HUNT
	}

	protected final IWorldEventListener<DefConChanged> level3DefconListener = new IWorldEventListener<DefConChanged>() {

		@Override
		public void notify(DefConChanged event) {
			switch (event.getNewDefCon()) {
				case 1:
					canLaunch = true;
					if (arrivalTime != -1) {
						arrivalTime = logic.getGameInfo().getGameTick();
					}
					logic.getLog().info("canLaunch " + fleet.getId());

					if (currentState == State.WAIT_FOR_DEFCON1) {
						currentState = State.LAUNCH_NUKES;
					}
					break;
				case 4:
					currentState = State.GOTO_ATTACK_POSITION;
					setMoveToTarget();
					logic.getLog()
							.info("goto attack position " + fleet.getId());
					break;
			}
		}
	};

	protected static final double TARGET_SELECTION_LIMIT = 50d;
	protected Fleet targetFleet;
	protected float lastTargetSelect = 0;


	public MixedFleetAI(DefConLocation target, Fleet fleet,
			ExampleBotLogicController logic) {
		super(logic, fleet.getId());

		this.fleet = fleet;

		logic.getWorldView().addEventListener(
				DefConChanged.class,
				level3DefconListener);

		enemyCityIds = logic.getGameInfo().getEnemyCityIds();
		this.originalLocation = new DefConLocation(fleet.getLocation());

		optimizeOriginalLocation();

		setTargetLocation(new DefConLocation(target));
		caculateBestMovementTarget();
	}

	protected void optimizeOriginalLocation() {
		DefConLocation cities_center = logic.getGameInfo()
				.getCitiesGravityCenter(logic.getGameInfo().getOwnTeamId());

		DefConLocation optimized = new DefConLocation(originalLocation);

		float curr_distance = Float.POSITIVE_INFINITY;
		do {
			optimized = getLogic().getFlagChecker()
					.traceFromTo(
							optimized,
							cities_center,
							BasicFlag.SEA,
							8d);

			if (optimized == null) {
				break;
			}

			curr_distance = logic.getGameInfo().getSailDistance(
					fleet.getLocation(),
					optimized);

		} while (Float.isInfinite(curr_distance));

		if (!Float.isInfinite(curr_distance)) {
			originalLocation = optimized;
		}

		act(new WhiteboardDraw(cities_center, originalLocation));
	}

	protected void setMoveToTarget() {
		moveFleet(currentTargetLocation);
	}

	protected void moveFleet(DefConLocation target) {

		if (target == null) {
			logic.getLog().info("moveFleet: null");
			return;
		}

		logic.getLog().info(
				"moveFleet: "
						+ fleet.getId() + " "
						+ fleet.getLocation() + " "
						+ target + " "
						+ logic.getGameInfo().getSailDistance(
								fleet.getLocation(),
								target));

		for (int unitId : fleet.getFleetMembers()) {
			logic.act(new SetMovementTarget(unitId,
					target));
		}
	}

	@Override
	public void update() {

		if (fleet.getFleetMembers().length == 0) {
			logic.getLog().info("no fleet members?!?! " + unitId);
		}

		switch (currentState) {
			case PRE_DEFCON4:
				break;
			case GOTO_ATTACK_POSITION: {
				runGotoAttackPosition();
				break;
			}
			case WAIT_FOR_DEFCON1:
				break;
			case LAUNCH_NUKES: {
				runLaunchNukes();
				break;
			}
			case HUNT: {
				runHunt();
				break;
			}
		}
	}

	protected void runGotoAttackPosition() {
		if (optimizedTargetLocation.getDistance(fleet.getLocation()) < 20d) {
			arrivalTime = logic.getGameInfo().getGameTick();
			logic.getLog().info("arrivalTime: " + arrivalTime);
			currentState = (canLaunch) ? State.LAUNCH_NUKES
					: State.WAIT_FOR_DEFCON1;
		}
	}

	protected void runLaunchNukes() {

		if (!hasSpareNukes()
				|| logic.getGameInfo().getGameTick() - arrivalTime > 750f) {
			logic.getLog().info("Hunt " + fleet.getId());
			currentState = State.HUNT;
		}

		if (!hasSpareLaunchableNukes())
			return;

		for (int unitId : fleet.getFleetMembers()) {

			if (logic.getGameInfo().getActionQueue(unitId).length > 20)
				continue;

			if (!nukeCapable(unitId)) {
				continue;
			}

			if (!inNukeState(unitId)) {
				toNukeState(unitId);
				continue;
			}

			DefConUnitObject<?> unit = (DefConUnitObject<?>) logic
					.getWorldView().get(WorldObjectId.get(unitId));

			int nukes = logic.getGameInfo().getNukeSupply(unitId);

			int tries = 10;
			while (nukes > 0 && tries > 0) {

				int cityId = enemyCityIds.get(logic.getRandom().nextInt(
						enemyCityIds.size()));


				if (!inNukeRange(unitId, cityId, unit.getType())) {
					--tries;

					continue;
				}
				//
				// logic.getLog().info(
				// "preparing nuking "
				// + nukes
				// + " " + unit.getType() + " "
				// + unit.getId());

				launchNuke(unitId, cityId);

				--nukes;
			}
		}
	}

	protected void runHunt() {

		final float tmp = logic.getGameInfo().getGameTick();

		for (int unitId : fleet.getFleetMembers()) {
			DefConUnitObject<?> unit = (DefConUnitObject<?>) logic
					.getWorldView().get(WorldObjectId.get(unitId));

			if (unit.getType() != null
					&& unit.getType() == UnitType.SUB
					&& unit.getState() != SubState.ACTIVE_SONAR) {
				act(new SetState(unitId, SubState.ACTIVE_SONAR.id));
			}

		}

		if (targetFleet != null) {

			if (tmp - lastTargetSelect < TARGET_SELECTION_LIMIT * 3)
				return;
			lastTargetSelect = tmp;

			logic.getLog().info("updating target: " + fleet.getId());

			if (targetFleet.isVisible()) {
				if (targetFleet.getLocation().getDistance(
						currentTargetLocation) > 35d) {

					refreshClosestEnemyFleet();
					if (targetFleet != null) {
						setAttackTarget(Integer.parseInt(targetFleet.getId()
								.getStringId()));
						currentTargetLocation = targetFleet.getLocation();
					}
					setMoveToTarget();
				}

			} else {


				refreshClosestEnemyFleet();
				if (targetFleet != null) {
					setAttackTarget((int) targetFleet.getId().getLongId());
					currentTargetLocation = targetFleet.getLocation();
				} else {
					currentTargetLocation = originalLocation;
				}
				setMoveToTarget();
			}

		} else {

			if (tmp - lastTargetSelect < TARGET_SELECTION_LIMIT)
				return;
			lastTargetSelect = tmp;

			logic.getLog().info("looking for: " + fleet.getId());

			refreshClosestEnemyFleet();

			if (targetFleet != null) {
				logic.getLog().info("Found new target: " + targetFleet.getId());
				setAttackTarget((int) targetFleet.getId()
						.getLongId());
				currentTargetLocation = targetFleet.getLocation();
				setMoveToTarget();
			} else {
				if (fleet.getLocation().getDistance(originalLocation) > 5d) {
					logic.getLog().info("Returning back");
					currentTargetLocation = originalLocation;
					setMoveToTarget();
				} else {
					logic.getLog().info("Already back");
				}
			}
		}
	}

	protected void setAttackTarget(int targetId) {
		for (int unitId : fleet.getFleetMembers()) {
			logic.act(new SetActionTarget(unitId,
					targetId, null));
		}
	}

	protected void refreshClosestEnemyFleet() {

		Fleet f = getClosestEnemyFleetWithUnits(new UnitType[] {
				UnitType.SUB, UnitType.CARRIER });

		if (f == null)
			f = getClosestEnemyFleet();

		if (f != null) {
			logic.getLog().info(
					"Targetted: " + f + " "
							+ arrayToString(f.getFleetMembers()));
			currentTargetLocation = f.getLocation();
		}

		targetFleet = f;
	}

	protected void caculateBestMovementTarget() {

		int enemyId = getClosestEnemyCityOwner(targetLocation);

		if (enemyId == -1)
			return;

		DefConLocation cities_center = logic.getGameInfo()
				.getCitiesGravityCenter(
						enemyId);

		currentTargetLocation = targetLocation;

		act(new WhiteboardDraw(cities_center,
				cities_center.add(new DefConLocation(0.5d, 0.5d))));
		act(new WhiteboardDraw(targetLocation,
				targetLocation.add(new DefConLocation(0.5d, 0.5d))));


		DefConLocation best_position_for_target = new DefConLocation(
				cities_center);
		float curr_distance;
		do {
			best_position_for_target = getLogic().getFlagChecker()
					.traceFromTo(
							targetLocation,
							best_position_for_target,
							BasicFlag.SEA,
							8d);

			curr_distance = logic.getGameInfo().getSailDistance(
					fleet.getLocation(),
					best_position_for_target);

			// logic.getLog().info(
			// "curr_distance: " + curr_distance + " "
			// + best_position_for_target + " " + fleet);

		} while (Float.isInfinite(curr_distance));

		double best_position_for_target_distance = best_position_for_target
				.getDistance2D(cities_center);
		// logic.getLog().info(
		// "best_for_target: " + best_position_for_target + " "
		// + best_position_for_target_distance);


		DefConLocation best_direct_position = new DefConLocation(cities_center);
		do {
			best_direct_position = getLogic().getFlagChecker()
					.traceFromTo(
							fleet.getLocation(),
							best_direct_position,
							BasicFlag.SEA,
							8d);

			if (best_direct_position == null) {
				break;
			}

			// logic.getLog().info(
			// "curr_distance: " + curr_distance + " "
			// + best_position_for_target + " " + fleet);
		} while (Float.isInfinite(logic.getGameInfo().getSailDistance(
				fleet.getLocation(),
				best_direct_position)));

		double best_direct_position_distance = (best_direct_position != null) ? best_direct_position
				.getDistance2D(cities_center)
				: Double.POSITIVE_INFINITY;


		if (best_direct_position_distance < best_position_for_target_distance) {
			currentTargetLocation = best_direct_position;
		} else {
			currentTargetLocation = best_position_for_target;
		}

		// logic.getLog().info(
		// "opt: " + logic.getGameInfo().getSailDistance(
		// fleet.getLocation(),
		// currentTargetLocation));

		optimizedTargetLocation = currentTargetLocation;

		act(new WhiteboardDraw(cities_center, currentTargetLocation));
	}

	protected int getClosestEnemyCityOwner(DefConLocation closestTo) {

		if (closestTo == null)
			return -1;

		double closest_distance = Double.POSITIVE_INFINITY;
		DefConLocation closest = new DefConLocation();
		int enemyId = -1;

		for (City city : logic.getWorldView().getAll(City.class).values()) {
			if (city.getTeamId() == logic.getGameInfo().getOwnTeamId())
				continue;

			double distance = closestTo.getDistance(city.getLocation());

			if (distance < closest_distance) {
				closest_distance = distance;
				closest = new DefConLocation(city.getLocation());
				enemyId = city.getTeamId();
			}
		}

		return enemyId;
	}

	protected String arrayToString(int[] array) {

		StringBuilder builder = new StringBuilder();
		builder.append("[");
		for (int element : array) {
			builder.append(element);
			builder.append(", ");
		}
		builder.append("]");

		return builder.toString();
	}

	protected boolean inNukeRange(int unitId, int cityId, UnitType type) {
		DefConLocation unitLocation = ((DefConUnitObject<?>) logic
				.getWorldView()
				.get(
						WorldObjectId.get(unitId))).getLocation();
		DefConLocation cityLocation = ((DefConViewableObject) logic
				.getWorldView()
				.get(
						WorldObjectId.get(cityId))).getLocation();

		return inNukeRange(unitLocation, cityLocation, type);
	}

	protected boolean inNukeRange(DefConLocation unit,
			DefConLocation target, UnitType type) {

		double dist = target.getDistance(unit);

		switch (type) {
			case SUB:
				return dist < logic.getGameInfo().getSubNukeRange();
			case CARRIER:
			case BOMBER:
				return dist < logic.getGameInfo().getBomberRange();
			case SILO:
				return true;
		}

		return false;
	}

	protected void launchNuke(int unitId, int targetId) {
		act(new SetActionTarget(unitId, targetId, null));
	}

	protected boolean nukeCapable(int unitId) {
		DefConUnitObject<?> object = (DefConUnitObject<?>) logic.getWorldView()
				.get(WorldObjectId.get(unitId));

		if (object == null)
			return false;

		if (object.getType() == null)
			return false;

		switch (object.getType()) {
			case CARRIER:
				return logic.getGameInfo().getNukeSupply(unitId) > 0
						&& logic.getGameInfo().getStateCount(
								unitId,
								CarrierState.BOMBER_LAUNCH) > 0;
			case SUB:
			case BOMBER:
			case SILO:
				return logic.getGameInfo().getNukeSupply(unitId) > 0;
			default:
				return false;
		}
	}

	protected boolean inNukeState(int unitId) {

		DefConUnitObject<?> object = (DefConUnitObject<?>) logic.getWorldView()
				.get(
						WorldObjectId.get(unitId));

		switch (object.getType()) {
			case CARRIER: {
				Carrier carrier = (Carrier) object;
				return carrier.getState() == CarrierState.BOMBER_LAUNCH;
			}
			case SUB: {
				Sub sub = (Sub) object;
				return sub.getState() == SubState.NUKE;
			}
		}

		return false;
	}

	protected boolean toNukeState(int unitId) {
		int nukes = logic.getGameInfo().getNukeSupply(unitId);

		if (nukes <= 0)
			return false;

		switch (logic.getGameInfo().getType(unitId)) {
			case SUB:
				act(new SetState(unitId, SubState.NUKE.getStateId()));
				break;
			case CARRIER:
				act(new SetState(unitId,
						CarrierState.BOMBER_LAUNCH.getStateId()));
				break;
		}

		return true;

	}

	protected boolean hasSpareNukes() {
		for (int unitId : fleet.getFleetMembers()) {
			int nukes = logic.getGameInfo().getNukeSupply(unitId);
			if (nukes > 0)
				return true;
		}
		return false;
	}

	protected boolean hasSpareLaunchableNukes() {
		for (int unitId : fleet.getFleetMembers()) {
			int nukes = logic.getGameInfo().getNukeSupply(unitId);


			if (logic.getGameInfo().getType(unitId) == UnitType.CARRIER) {
				nukes -= logic.getGameInfo().getStateCount(
						unitId,
						CarrierState.BOMBER_LAUNCH);
			}

			if (nukes > 0)
				return true;


		}
		return false;
	}

	public final Fleet getFleet() {
		return fleet;
	}

	public final ExampleBotLogicController getLogic() {
		return (ExampleBotLogicController) logic;
	}

	@Override
	public void setTargetLocation(DefConLocation target) {
		targetLocation = target;
	}

	/**
	 * Returns the closest visible enemy fleet.
	 * 
	 * @return closest visible enemy fleet
	 */
	public Fleet getClosestEnemyFleet() {
		return getClosestEnemyFleetWithUnit(null);
	}

	/**
	 * Returns the closest visible enemy fleet with given unit type.
	 * 
	 * @param type
	 *            unitetype the enemy fleet has to contain
	 * 
	 * @return closest visible enemy fleet with given unitetype
	 */
	public Fleet getClosestEnemyFleetWithUnit(UnitType type) {

		return getClosestEnemyFleetWithUnits(new UnitType[] { type });
	}

	/**
	 * Returns the closest visible enemy fleet with given unit type.
	 * 
	 * @param type
	 *            unitetype the enemy fleet has to contain
	 * 
	 * @return closest visible enemy fleet with given unitetype
	 */
	public Fleet getClosestEnemyFleetWithUnits(UnitType[] types) {
		double closest_dist = Double.POSITIVE_INFINITY;
		Fleet closest_fleet = null;

		for (int enemyId : logic.getGameInfo().getEnemyTeamIds()) {
			for (Fleet enemy_fleet : getLogic().getFleetsManager()
					.getEnemyFleets(enemyId)) {

				if (enemy_fleet.getTeamId() == logic.getGameInfo()
						.getOwnTeamId())
					continue;

				if (!enemy_fleet.isVisible())
					continue;

				if (types != null && types.length > 0) {
					boolean found = false;
					for (int unitId : enemy_fleet.getFleetMembers()) {
						DefConUnitObject<?> unit = (DefConUnitObject<?>) logic
								.getWorldView().get(WorldObjectId.get(unitId));

						if (unit == null || !unit.isVisible())
							continue;

						if (unit.getType() != null
								&& arrayContains(types, unit.getType())) {
							found = true;
							break;
						}
					}
					if (!found)
						continue;
				}

				double dist = logic.getGameInfo().getSailDistance(
						fleet.getLocation(),
						enemy_fleet.getLocation());

				if (dist < closest_dist) {
					closest_fleet = enemy_fleet;
					closest_dist = dist;
				}
			}
		}

		return closest_fleet;
	}

	@Override
	public void dispose() {
		super.dispose();
		logic.getWorldView().removeListener(level3DefconListener);
	}

	protected <T> boolean arrayContains(T[] array, T element) {
		return arrayIndex(array, element) != -1;
	}

	protected <T> int arrayIndex(T[] array, T element) {

		if (element == null)
			return -1;

		for (int i = 0; i < array.length; ++i) {
			if (array[i] != null && array[i].equals(element))
				return i;
		}
		return -1;
	}

	@Override
	public int[] getFleetMembers() {
		return fleet.getFleetMembers();
	}

	@Override
	public DefConLocation getLocation() {
		return fleet.getLocation();
	}

}
