package cz.cuni.amis.pogamut.defcon.example.javabot;

import java.util.Map;

import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.defcon.agent.impl.DefConAgentLogicController;
import cz.cuni.amis.pogamut.defcon.ai.buildingai.AbstractBuildingAI;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.SetActionTarget;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.SetState;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.AirBase;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConChanged;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConViewableObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Sub;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;
import cz.cuni.amis.pogamut.defcon.consts.state.AirBaseState;
import cz.cuni.amis.pogamut.defcon.consts.state.SubState;

/**
 * AirBase AI.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class AirBaseAI extends AbstractBuildingAI<AirBase> {

	private final static UnitType[] preNukePriorities = {
			UnitType.BOMBER,
			UnitType.CARRIER, UnitType.FIGHTER, UnitType.SUB,
			UnitType.BATTLE_SHIP };
	private final static UnitType[] nukePriorities = {
			UnitType.CITY, UnitType.BOMBER,
			UnitType.CARRIER, UnitType.FIGHTER,
			UnitType.SUB,
			UnitType.AIR_BASE,
			UnitType.BATTLE_SHIP,
			UnitType.NUKE };

	private UnitType[] priorities = preNukePriorities;

	private boolean canLaunch = false;

	private final double squareBomberRange;
	private final double squareFighterRange;

	private final IWorldEventListener<DefConChanged> defconListener = new IWorldEventListener<DefConChanged>() {

		@Override
		public void notify(DefConChanged event) {
			if (event.getNewDefCon() == 1) {
				canLaunch = true;
				priorities = nukePriorities;
			}
		}
	};

	public AirBaseAI(AirBase airbase, DefConAgentLogicController<?> logic) {
		super(airbase, logic);
		logic.getWorldView().addEventListener(
				DefConChanged.class,
				defconListener);

		squareBomberRange = logic.getGameInfo().getBomberRange()
				* logic.getGameInfo().getBomberRange();
		squareFighterRange = logic.getGameInfo().getFighterRange()
				* logic.getGameInfo().getFighterRange();
	}

	@Override
	public void update() {

		if (logic.getGameInfo().getStateTimer(unitId) > 0) {
			return;
		}


		int bombers = logic.getGameInfo().getStateCount(
				unitId,
				AirBaseState.BOMBER_LAUNCH);
		int fighters = logic.getGameInfo().getStateCount(
				unitId,
				AirBaseState.FIGHTER_LAUNCH);

		if (bombers > 0 || fighters > 0) {

			// testing if periodically iteration through ALL units is ok
			// if not, then use random generator to set a probability of
			// such an update

			DefConViewableObject target = null;
			// lesser is better
			int targets_priority = priorities.length;

			Map<WorldObjectId, DefConViewableObject> possible_targets = logic
					.getWorldView()
					.getAllVisible(DefConViewableObject.class);

			Object[] keys = possible_targets.keySet()
					.toArray();

			if (keys == null || keys.length == 0)
				return;

			for (int tries = 0; tries < 30; ++tries) {

				DefConViewableObject unit = possible_targets
						.get(keys[logic.getRandom().nextInt(keys.length)]);

				if (unit.getType() == UnitType.SUB
						&& ((Sub) unit).getState() == SubState.PASSIVE_SONAR) {
					continue;
				}

				if (!unit.isVisible())
					continue;

				if (unit.getTeamId() == logic.getGameInfo().getOwnTeamId())
					continue;

				if (unit.getLocation() == null
						|| (unit.getLocation().x == 0 && unit.getLocation().y == 0)) {
					logic.getLog().info("location is null");
					continue;
				}

				if (!canLaunch && isNukeTarget(unit.getType()))
					continue;

				int priority = arrayIndex(priorities, unit.getType());

				if (priority < 0)
					continue;

				double square_distance = building.getLocation()
						.getDistanceSquare(
								unit.getLocation());

				if (canLaunch && isBomberTarget(unit.getType())
						&& (bombers == 0
						|| square_distance > squareBomberRange)) {

					continue;
				}

				if (isFighterTarget(unit.getType()) && (fighters == 0
						|| square_distance > squareFighterRange)) {
					continue;
				}

				if (priority < targets_priority) {
					target = unit;
					targets_priority = priority;

				} else if (priority == targets_priority) {
					float rand = logic.getRandom().nextFloat();
					if (rand > 0.15f) {
						target = unit;
					}
				}
			}

			if (target != null) {
				launchAppropriateAircraft(
						target.getType(),
						Integer.parseInt(target.getId().getStringId()));
			}

		} else {
			
		}
	}

	private boolean isNukeTarget(UnitType unitType) {
		switch (unitType) {
			case CITY:
			case AIR_BASE:
			case SILO:
			case RADAR:
				return true;
		}

		return false;
	}

	private boolean isFighterTarget(UnitType unitType) {
		switch (unitType) {
			case FIGHTER:
			case BOMBER:
			case BATTLE_SHIP:
			case CARRIER:
				return true;
		}

		return false;
	}

	private boolean isBomberTarget(UnitType unitType) {
		switch (unitType) {
			case AIR_BASE:
			case RADAR:
			case SILO:
			case CITY:
				return true;
		}

		return false;
	}

	private void launchAppropriateAircraft(UnitType target, int targetId) {
		// logic.getLog()
		// .info(
		// "Launching aircraft: " + unitId + " " + targetId + " "
		// + target);
		switch (target) {
			case AIR_BASE:
			case BATTLE_SHIP:
			case CARRIER:
			case RADAR:
			case SILO:
			case SUB:
			case CITY:
				if (building.getState() != AirBaseState.BOMBER_LAUNCH)
					act(new SetState(unitId, AirBaseState.BOMBER_LAUNCH.id));
				act(new SetActionTarget(unitId, targetId, null));
				break;
			case FIGHTER:
			case BOMBER:
			case NUKE:
				if (building.getState() != AirBaseState.FIGHTER_LAUNCH)
					act(new SetState(unitId, AirBaseState.FIGHTER_LAUNCH.id));
				act(new SetActionTarget(unitId, targetId, null));
				break;
		}
	}

	private <T> int arrayIndex(T[] array, T element) {

		if (element == null)
			return -1;

		for (int i = 0; i < array.length; ++i) {
			if (array[i] != null && array[i].equals(element))
				return i;
		}
		return -1;
	}

	@Override
	public void dispose() {
		super.dispose();
		logic.getWorldView().removeListener(defconListener);
	}
}
