/*
 * Copyright (C) 2012 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.spyvsspy.mapgenerator;

import cz.cuni.amis.utils.collections.MyCollections;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import org.apache.commons.lang.ArrayUtils;

/**
 *
 * @author Martin Cerny
 */
public class RandomMapGenerator extends AbstractRandomMapGenerator {

    int numKeyPoints;

    double doorOpenDensity;

    double doorCloseDensity;

    /**
     * 
     * @param width
     * @param height
     * @param numKeyPoints number of key points of the desired solution path
     * @param doorOpenDensity how many doors can be (on average) opened with one button - this is taken to maximum number of buttons per rectangular level of this size
     * @param doorCloseDensity how many doors can be (on average) closed with one button - this is taken to maximum number of buttons per rectangular level of this size
     */
    public RandomMapGenerator(long seed, int width, int height, int numKeyPoints, double doorOpenDensity, double doorCloseDensity) {
        super(seed,width,height);
        this.numKeyPoints = numKeyPoints;
        this.doorOpenDensity = doorOpenDensity;
        this.doorCloseDensity = doorCloseDensity;
    }

    /**
     * Creates a random map. A path between 0,0 and width -1, height - 1 will exist
     * @return 
     */
    public Collection<ButtonDefinition> createRandomMapDefinition() {
        int numDoorOpens = 0;
        int numDoorClose = 0;

        int desiredOpenDoorCount = (int) (getMaximumNumberOfButtons() * doorOpenDensity);
        int desiredCloseDoorCount = (int) (getMaximumNumberOfButtons() * doorCloseDensity);

        buttonDefinitions = new HashMap<ButtonLocation, ButtonDefinition>();

        //TODO poresit, abych vynutil opakovani lokaci (tj. kdyz musim nekam dvakrat, tak nemuzu obe tlacitka, co tam mam zmacknout naraz

        //Generate guaranteed solution path
        List<Point> path = new ArrayList<Point>();
        path.add(new Point(0, 0));
        for (int i = 0; i < numKeyPoints; i++) {
            int keyPointX = rnd.nextInt(width);
            int keyPointY = rnd.nextInt(height);
            while (keyPointX == width - 1 && keyPointY == height - 1) {
                keyPointX = rnd.nextInt(width);
                keyPointY = rnd.nextInt(height);
            }

            //System.out.println("KeyPoint: " + keyPointX + ", " + keyPointY);
            appendPathToKeyPoint(path, keyPointX, keyPointY);
        }
        appendPathToKeyPoint(path, width - 1, height - 1);

        for(Point p : path){
            System.out.println(p);
        }

        //indices of button locations on guaranteed path that open the respective path corridor
        Integer ensureIndices[] = new Integer[path.size() - 1];
        
        //Create open buttons to make guaranteed solution path feasible
        //Dont need to iterate down to 0 - starting position is always reachable
        for(int i = path.size() -1; i > 0; i--){
            /* 
             * index of location where I open door to this place
             * Biased towards later indices, so that the interactions are not crowded at the beginning of
             * the path.
             */
            int ensureIndex;
            if(i > 1) {
                ensureIndex = (int)Math.sqrt(rnd.nextInt((i * i) - 1));
            }  else {
                ensureIndex = 0;
            }
            ensureIndices[i - 1] = ensureIndex;
            
            addButtonInteractionToLocation(path.get(ensureIndex), path.get(i-1), path.get(i), true);
            numDoorOpens++;
        }
        
        if(numDoorOpens > desiredOpenDoorCount){
            double newDensity = ((double)numDoorOpens) / getMaximumNumberOfButtons();
            System.out.println("Needed more than desired number of interactions to reach the target. The doorOpenDensity is " + newDensity + ", should have been " + doorOpenDensity);
        }
        
        //create more random door opens
        while(numDoorOpens < desiredOpenDoorCount){
            boolean addedNew = false;
            while (!addedNew){
                Point randomTo = getRandomPoint();
                Point randomFrom = MyCollections.getRandom(getNeighbouringPoints(randomTo));
                Point buttonLocation = getRandomPoint();
                addedNew = addButtonInteractionToLocation(buttonLocation, randomFrom, randomTo, true);
                //iterate until we find an interaction that was not added - this should not be very difficult
            }
            numDoorOpens++;                        
        }

        //create more, random door closing.
        //check, whether such an interaction won't interrupt with the guaranteed solution path
        while(numDoorClose < desiredCloseDoorCount){
            boolean addedNew = false;
            while (!addedNew){
                Point randomTo = getRandomPoint();
                Point randomFrom = MyCollections.getRandom(getNeighbouringPoints(randomTo));
                Point buttonLocation = getRandomPoint();
                if(threatensGuaranteedPath(path, ensureIndices, buttonLocation, randomFrom, randomTo)){
                    continue;
                }
                addedNew = addButtonInteractionToLocation(buttonLocation, randomFrom, randomTo, false);
                //iterate until we find an interaction that was not added - this should not be very difficult
            }
            numDoorClose++;                        
        }
        
        

        return buttonDefinitions.values();
    }
    
    protected boolean threatensGuaranteedPath(List<Point> path, Integer[] ensureIndices, Point buttonLocation, Point from, Point to){
        if(!path.contains(to) || !path.contains(from) || !path.contains(buttonLocation)){
            return false;            
        }
        for(int i = 0; i < path.size(); i++){
            if(path.get(i).equals(buttonLocation)){
                //i is index of button location in path
                if(ArrayUtils.contains(ensureIndices, i)){
                    //there is problem only if the button ensures something else, 
                    for(int j = path.size() - 1; j > 0; j--){
                        if( (path.get(j).equals(to) && path.get(j - 1).equals(from))
                                || path.get(j).equals(from) && path.get(j-1).equals(to)){
                            //and if the door is not opened later on
                            if(ensureIndices[j - 1] <= i){
                                return true;
                            }
                        }
                    }
                }
            }
        }
        return false;
    }

    protected void appendPathToKeyPoint(List<Point> targetPath, int keyPointX, int keyPointY) {
        Point lastPathPoint = targetPath.get(targetPath.size() - 1);
        Point targetPoint = new Point(keyPointX, keyPointY);
        while (!lastPathPoint.equals(targetPoint)) {
            int distanceX = Math.abs(lastPathPoint.x - keyPointX);
            int distanceY = Math.abs(lastPathPoint.y - keyPointY);
            boolean moveX;
            if (distanceX == 0) {
                moveX = false;
            } else if (distanceY == 0) {
                moveX = true;
            } else {
                int totalDistance = distanceX + distanceY;
                double moveXProbability = (double) distanceX / (double) totalDistance;
                moveX = rnd.nextDouble() < moveXProbability;
            }

            int nextPointX = lastPathPoint.x;
            int nextPointY = lastPathPoint.y;
            if (moveX) {
                if (keyPointX > lastPathPoint.x) {
                    nextPointX++;
                } else {
                    nextPointX--;
                }
            } else {
                if (keyPointY > lastPathPoint.y) {
                    nextPointY++;
                } else {
                    nextPointY--;
                }
            }
            lastPathPoint = new Point(nextPointX, nextPointY);
            targetPath.add(lastPathPoint);
        }
    }

}
