package nl.tudelft.pogamut.ut2004.agent.module.shooting.weapon;

import nl.tudelft.pogamut.ut2004.agent.module.shooting.AbstractWeaponShooting;
import nl.tudelft.pogamut.ut2004.agent.module.shooting.WeaponShooting;
import nl.tudelft.pogamut.ut2004.agent.module.shooting.util.FacingUtil;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;

/**
 * <p>
 * Module to efficient work with the link gun. Will power up team mates when
 * possible.
 * </p>
 * 
 * <p>
 * To power up a team mate the
 * {@link ShockRifleShooting#shoot(WeaponPref, ILocated)} must be called with a
 * target that is a friendly player. In which case the weaponPref will be
 * ignored. The module will only shoot at team mates who are in less then
 * {@link LinkGunShooting#LINK_GUN_SEC_MAX_RANGE} UT units away. To conserve
 * ammo this shooting will not shoot at friendly players when they are not in
 * range.
 * </p>
 * 
 * <p>
 * When facing a shielded player, the module will switch to secondary fire mode
 * if the target is within secondary range or stop shooting all together.
 * </p>
 * 
 * @author mpkorstanje
 * 
 */
public class LinkGunShooting extends AbstractWeaponShooting implements WeaponShooting {

	/**
	 * Angle at which shield gun can deflect projectiles.
	 */
	protected static final int SHIELD_GUN_DEFLECT_ANGLE = 90;

	/**
	 * Practical experiments show that the link guns secondary fire has a range
	 * of 1150UT units. When attached to a friendly player, the range is ~1500UT
	 * unit. Since we can't test attachment we use 1150UT units.
	 */
	protected static final int LINK_GUN_SEC_MAX_RANGE = 1150;
	/**
	 * Secondary mode preference.
	 */
	protected static final WeaponPref LINK_GUN_SECONDARY = new WeaponPref(ItemType.LINK_GUN, false);;
	/**
	 * Primary mode preference.
	 */
	protected static final WeaponPref LINK_GUN_PRIMARY = new WeaponPref(ItemType.LINK_GUN, true);;


	public LinkGunShooting(UT2004Bot<?, ?, ?> bot, AgentInfo info, ImprovedShooting shoot, Weaponry weaponry) {
		super(bot, info, shoot, weaponry);
	}

	@Override
	protected void shoot() {

		// Wrong weapon, wait up.
		if (!isWeaponReady()) {
			return;
		}
		
		//No target, no shoot.
		if(!hasTarget()){
			shoot.stopShooting();
			return;
		}


		// Not a player. Blast away.
		if (!(target instanceof Player)) {
			shoot.shoot(weaponPref, target);
			return;
		}

		Player player = (Player) target;

		// Target not visible, hold fire.
		if (!player.isVisible()) {
			shoot.stopShooting();
			return;
		}

		// Assist friend, or hold fire if out of range, or shoot foe.
		if (info.isFriend(player)) {
			shootFriend(player);
			return;
		}

		if(!info.isFacing(target)){
			shoot.stopShooting();
			return;
		}

		
		boolean targetFacing = FacingUtil.isFacing(player, agent, SHIELD_GUN_DEFLECT_ANGLE);
		boolean targetShieldGun = ItemType.SHIELD_GUN.equals(ItemType.getItemType(player.getWeapon()));
		boolean targetShielded = player.getFiring() == 2;
		boolean targetCanDeflect = targetShieldGun && targetFacing && targetShielded;
		boolean closeEnough = info.getLocation().getDistance(player.getLocation()) < LINK_GUN_SEC_MAX_RANGE;

		
		if (targetCanDeflect && closeEnough) {
			shoot.shoot(LINK_GUN_SECONDARY, target);
		} else if (!targetCanDeflect && !closeEnough) {
			shoot.shoot(LINK_GUN_PRIMARY, target);
		} else if (!targetCanDeflect){
			shoot.shoot(weaponPref, target);
		} else {
			shoot.stopShooting();
		}

	}

	protected void shootFriend(Player player) {
		if (canAssist(player) && info.isFacing(player)) {
			shoot.shoot(LINK_GUN_SECONDARY, player);
		} else {
			shoot.stopShooting();
		}
	}

	protected boolean canAssist(Player player) {
		if (player == null) {
			return false;
		}

		boolean friend = info.isFriend(player);
		boolean hasLinkGun = ItemType.LINK_GUN.equals(ItemType.getItemType(player.getWeapon()));
		boolean closeEnough = info.getLocation().getDistance(player.getLocation()) < LINK_GUN_SEC_MAX_RANGE;

		return friend && hasLinkGun && closeEnough;
	}

	/**
	 * Primary fire mode is preferred for the link gun.
	 */
	@Override
	protected WeaponPref getDefaultWeaponPref() {
		return LINK_GUN_PRIMARY;
	}


}
