package nl.tudelft.pogamut.ut2004.agent.module.shooting;

import nl.tudelft.pogamut.ut2004.agent.module.shooting.util.FocusProvider;
import cz.cuni.amis.pogamut.base.agent.module.SensorModule;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;

/**
 * Base class for all shootings. Handles setting of target and weapon prefs and
 * automates calling of {@link AbstractWeaponShooting#shoot()}.
 * 
 * @author mpkorstanje
 * 
 */
@SuppressWarnings("rawtypes")
public abstract class AbstractWeaponShooting extends SensorModule<UT2004Bot> implements WeaponShooting {

	protected static final double FACING_ANGLE = 25.0;
	/**
	 * Half of height of player, aim this much below player to shoot at feet.
	 */
	protected static final Location BELOW_PLAYER_OFFSET = new Location(0, 0, 44);
	/**
	 * Height of player, aim this much above player to shoot around shield.
	 */
	protected static final Location ABOVE_PLAYER_OFFSET = new Location(0, 0, 88);
	/**
	 * Our target to shoot. May be null, in which case we may charge a weapon.
	 */
	protected ILocated target = null;
	/**
	 * Weapon pref indicates which firing mode to use.
	 */
	protected WeaponPref weaponPref;
	/**
	 * Reference to {@link ImprovedShooting} module.
	 */
	protected ImprovedShooting shoot;
	/**
	 * Reference to {@link Weaponry} module.
	 */

	protected Weaponry weaponry;
	/**
	 * Reference to {@link AgentInfo} module.
	 */

	protected AgentInfo info;
	/**
	 * If this shooting has been activated.
	 */
	protected boolean active = false;

	/**
	 * Where the shooting would like the bot to look.
	 */
	protected FocusProvider focus = new FocusProvider();

	/**
	 * Creates an abstract WeaponShooting.
	 * 
	 * @param agent
	 *            the bot that does the shooting.
	 * @param info
	 *            the info module for the bot.
	 * @param shoot
	 *            the shooting module for the bot.
	 * @param weaponry
	 *            the weaponry module for the bot.
	 */
	public AbstractWeaponShooting(UT2004Bot<?, ?, ?> agent, AgentInfo info, ImprovedShooting shoot, Weaponry weaponry) {
		super(agent);
		this.shoot = shoot;
		this.info = info;
		this.weaponry = weaponry;
		this.endMessageListener = new EndMessageListener(worldView);
	}

	/**
	 * Set target to shoot at, actual shooting is deferred to next end message.<br>
	 * 
	 * Callers should ensure that weapon used by is available and has sufficient
	 * ammunition.
	 * 
	 * @param player
	 */
	@Override
	public void shoot(WeaponPref weaponPref, ILocated target) {
		if (weaponPref == null || !weaponPref.getWeapon().equals(getWeaponType())) {
			throw new IllegalArgumentException(
					"Weapon pref must be correct for the weapon associated with this shooting");
		}

		this.active = true;
		this.target = target;
		this.focus.setFocus(target);
		this.weaponPref = weaponPref;
	}

	/**
	 * @return the default weapon preference.
	 */
	protected abstract WeaponPref getDefaultWeaponPref();

	/**
	 * Stops the shooting.
	 */
	public void stopShoot() {
		this.active = false;
		this.target = null;
		this.focus.clearFocus();
		shoot.stopShooting();
	}

	/**
	 * Sets where the shooting would like to look.
	 * 
	 * @param focus
	 *            where we want to look.
	 */
	protected void setFocus(ILocated focus) {
		this.focus.setFocus(focus);
	}

	/**
	 * Where the shooting would like the agent to look.
	 */
	public ILocated getFocus() {
		return focus;
	}

	/**
	 * 
	 * @return true iff we have target.
	 */
	public boolean hasTarget() {
		return target != null;
	}

	/**
	 * The weapon type this module can do the shooting for.
	 * 
	 * @return the weapon type we'll shoot with.
	 */
	public ItemType getWeaponType() {
		return getDefaultWeaponPref().getWeapon();
	}

	/**
	 * @return true if the shooting has been activated.
	 */
	@Override
	public boolean isActive() {
		return active;
	}

	/**
	 * @return if we have the {@link WeaponShooting#getWeaponType()}.
	 */
	protected boolean isWeaponReady() {
		// We should have the link gun ready.
		if (weaponry.getCurrentWeapon() == null || !weaponry.getCurrentWeapon().getType().equals(getWeaponType())) {
			weaponry.changeWeapon(getWeaponType());
			return false;
		}

		return true;
	}

	/**
	 * <p>
	 * Shoot will be called after every end message.
	 * </p>
	 * 
	 * <p>
	 * Implementing subclasses should make a best effort attempt to shoot the
	 * given target using the given weapon preference if possible. Subclasses
	 * should take care to note that a target may not always be present. e.g.
	 * target == null.
	 * </p>
	 * 
	 */
	protected abstract void shoot();

	/**
	 * {@link EndMessage} listener.
	 */
	private class EndMessageListener implements IWorldEventListener<EndMessage> {
		@Override
		public void notify(EndMessage event) {
			if (isActive())
				shoot();
		}

		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * 
		 * @param worldView
		 *            WorldView object to listen to.
		 */
		public EndMessageListener(IWorldView worldView) {
			worldView.addEventListener(EndMessage.class, this);
		}
	}

	/** {@link EndMessage} listener */
	protected EndMessageListener endMessageListener;

}