package cz.cuni.amis.pogamut.base.utils.math;

import java.util.Collection;

import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.utils.IFilter;

/**
 *
 * @author ik
 */
public class DistanceUtils {
	
    /**
     * Returns the nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearest(Collection<T> locations, ILocated target) {
        T nearest = null;
        Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (l.getLocation() == null) continue;
            d = l.getLocation().getDistance(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }
    
    /**
     * Returns the nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearestFiltered(Collection<T> locations, ILocated target, IFilter<T> filter) {
        Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
        T nearest = null;        
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (l.getLocation() == null) continue;
        	if (!filter.isAccepted(l)) continue;
            d = l.getLocation().getDistance(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }

    /**
     * Returns the second nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {
        	if (l.getLocation() == null) continue;
            double distance = Location.getDistance(l.getLocation(), targetLoc);
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }
    
    /**
     * Returns the nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends IViewable> T getNearestVisible(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T nearest = null;
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (!l.isVisible()) continue;
        	ILocated loc = (ILocated)l;
        	if (loc.getLocation() == null) continue;
            d = loc.getLocation().getDistance(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }

    /**
     * Returns the second nearest object to 'target'.
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends IViewable> T getSecondNearestVisible(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {        
            if (!l.isVisible()) continue;
        	ILocated loc = (ILocated)l;
        	if (loc.getLocation() == null) continue;
        	double distance = Location.getDistance(loc.getLocation(), targetLoc);
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }
    
    /**
     * Returns the nearest object to 'target' (ignoring 'z' coordinate).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearest2D(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T nearest = null;
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (l.getLocation() == null) continue;
            d = l.getLocation().getDistance2D(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }
    
    /**
     * Returns the nearest object to 'target' (ignoring 'z' coordinate).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getNearestFiltered2D(Collection<T> locations, ILocated target, IFilter<T> filter) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T nearest = null;
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (!filter.isAccepted(l)) continue;
        	if (l.getLocation() == null) continue;
            d = l.getLocation().getDistance2D(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }

    /**
     * Returns the second nearest object to 'target' (ignoring 'z' coordinate).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends ILocated> T getSecondNearest2D(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {
        	if (l.getLocation() == null) continue;
            double distance = Location.getDistance2D(l.getLocation(), targetLoc);
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }
    
    /**
     * Returns the nearest object to 'target' (ignoring 'z' coordinate).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends IViewable> T getNearestVisible2D(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T nearest = null;
        double minDistance = Double.MAX_VALUE;
        double d;
        for(T l : locations) {
        	if (!l.isVisible()) continue;
        	ILocated loc = (ILocated)l;
        	if (loc.getLocation() == null) continue;
            d = loc.getLocation().getDistance2D(targetLoc);
            if(d < minDistance) {
                minDistance = d;
                nearest = l;
            }
        }
        return nearest;
    }

    /**
     * Returns the second nearest object to 'target' (ignoring 'z' coordinate).
     * <p><p>
     * WARNING: O(n) complexity!
     * 
     * @param <T>
     * @param locations must be objects implementing {@link IViewable} as well as {@link ILocated} (so {@link Item} or {@link Player} is usable)
     * @param target
     * @return nearest object from collection of objects
     */
    public static <T extends IViewable> T getSecondNearestVisible2D(Collection<T> locations, ILocated target) {
    	Location targetLoc = target.getLocation();
        if (targetLoc == null) return null;
    	T secondNearest = null;
        T nearest = null;
        double closestDistance = Double.MAX_VALUE;
        double secondClosestDistance = Double.MAX_VALUE;

        for (T l : locations) {        
            if (!l.isVisible()) continue;
        	ILocated loc = (ILocated)l;
        	if (loc.getLocation() == null) continue;
        	double distance = Location.getDistance2D(loc.getLocation(), targetLoc);
            if (distance < closestDistance) {
                secondClosestDistance = closestDistance;
                secondNearest = nearest;

                closestDistance = distance;
                nearest = l;
            } else {
                if(distance < secondClosestDistance) {
                    secondClosestDistance = distance;
                    secondNearest = l;
                }
            }
        }
        return secondNearest;
    }

}
