/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.cuni.pogamut.posh.palette.external;

import cz.cuni.amis.pogamut.sposh.JavaBehaviour;
import cz.cuni.amis.pogamut.sposh.SPOSHAction;
import cz.cuni.amis.pogamut.sposh.SPOSHSense;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.jar.Attributes;
import java.util.logging.Logger;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.spi.java.classpath.ClassPathProvider;
import org.openide.loaders.DataObject;
import org.openide.util.Exceptions;

/**
 * External loader of behaviours from external {@code JAR}s.
 * @author Honza
 */
final public class ExternalBehaviourLoader {

    private static Logger log = Logger.getLogger(ExternalBehaviourLoader.class.getName());
    private DataObject dataObj;
    private Map<Class, BehaviourPrimitives> behaviourInfo = new HashMap<Class, BehaviourPrimitives>();

    public ExternalBehaviourLoader(DataObject dataObj) {
        this.dataObj = dataObj;

    }

    /**
     * Returmn unmodifiable behaviour info, create if necessary.
     * @return
     */
    public synchronized Map<Class, BehaviourPrimitives> getBehaviourInfo() {
        if (behaviourInfo == null) {
            behaviourInfo = createBehaviourInfo();
        }
        return behaviourInfo;
    }

    public synchronized Map<Class, BehaviourPrimitives> createBehaviourInfo() {
        Project project = FileOwnerQuery.getOwner(dataObj.getPrimaryFile());
        if (project == null) {
            throw new IllegalStateException("File " + dataObj.getPrimaryFile().getNameExt() + " has no project owner.");
        }

        ClassPathProvider classPathResult = project.getLookup().lookup(ClassPathProvider.class);
        if (classPathResult == null) {
            throw new IllegalArgumentException("Project " + project.toString() + " has no ClassPathProvider.");
        }

        List<Class> behaviourClasses = getAllBehaviourClasses(classPathResult, ClassPath.EXECUTE);

        for (Class behaviourClass : behaviourClasses) {
            behaviourInfo.put(behaviourClass, new BehaviourPrimitives(behaviourClass));
        }
        return Collections.unmodifiableMap(behaviourInfo);
    }

    /**
     * Print info about behaviour classses and its senses and actions to the log.
     */
    protected synchronized void printInfo() {
        log.info("Behaviour info:");
        for (Entry<Class, BehaviourPrimitives> entry : behaviourInfo.entrySet()) {
            log.info(entry.getKey().getCanonicalName() + ": " + entry.getValue());
        }
    }

    /**
     * Get list of all behaviour classes from all jars on the classpath of the project.
     * Jar has behaviour class, when it has X-BEHAVIOUR attribute in its element.
     * If something is wrong with the jar (can't read, class is not there...), skip it.
     *
     * @param cpp ClassPathProvider for project we are interested in
     * @param type What type of classpath sre we looking for. Values from ClassPath.*
     * @return List of behaviour classes
     */
    private List<Class> getAllBehaviourClasses(ClassPathProvider cpp, String type) {
        ClassPath classPath = cpp.findClassPath(dataObj.getPrimaryFile(), type);

        List<Class> behaviourClasses = new ArrayList<Class>();
        // Get valid behaviour classes from all valid jars in the classpath
        for (ClassPath.Entry entry : classPath.entries()) {
            try {
                behaviourClasses.add(getBehaviourClass(entry));
            } catch (IllegalArgumentException ex) {
                log.info(ex.getMessage());
            } catch (Exception ex) {
                log.warning(ex.getMessage());
                Exceptions.printStackTrace(ex);
            }
        }

        return behaviourClasses;
    }

    private Class getBehaviourClass(ClassPath.Entry entry) throws IOException, ClassNotFoundException {
        if (!entry.isValid()) {
            throw new IllegalArgumentException("Classpath entry (" + entry + ") is not valid. Skipping.");
        }
        if (!isJar(entry.getURL())) {
            throw new IllegalArgumentException("Classpath entry (" + entry + ") is not jar. Skipping.");
        }

        
        // Get clasLoader for the jar file.
        URL url = entry.getURL();
        URLClassLoader jarFileLoader = new URLClassLoader(new URL[]{url}, JavaBehaviour.class.getClassLoader());
        // Get manifest from the jar



        //Manifest manifest = new Manifest(jarFileLoader.getResourceAsStream("META-INF/MANIFEST.MF"));
        //Manifest manifest = new Manifest(entry.getRoot().getInputStream());
        // new JarFile().getManifest()
        JarURLConnection jarUrl = (JarURLConnection) url.openConnection();


        // Check X-BEHAVIOUR attribute in the manifest
        Attributes mfAttributes = jarUrl.getMainAttributes();

        String behaviourClassName = mfAttributes.getValue(new Attributes.Name("X-BEHAVIOUR"));
        if (behaviourClassName == null) {
            throw new IllegalArgumentException("Classpath entry (" + entry + ") manifest doesn't contain X-BEHAVIOUR entry. Skipping.");
        }

        // Get class behaviour from the jar
        return jarFileLoader.loadClass(behaviourClassName);
    }

    /**
     * Tests if passed url is local jar.
     * @param url url to be tested. 
     * @return true if it is jar, false otherwise. Checking only protocol part of url.
     */
    private boolean isJar(URL url) {
        if ("jar".equals(url.getProtocol())) {
            return true;
        }
        return false;
    }

    /**
     * Class storing the info about senses and actions of the behaqviour class.
     */
    final public static class BehaviourPrimitives {

        final private List<Method> actions;
        final private List<Method> senses;

        public BehaviourPrimitives(Class behaviourClass) {
            this.actions = getActions(behaviourClass);
            this.senses = getSenses(behaviourClass);
        }

        public BehaviourPrimitives(Collection<Method> actions, Collection<Method> senses) {
            this.actions = new ArrayList<Method>(actions);
            this.senses = new ArrayList<Method>(senses);
        }

        private List<Method> getActions(Class behaviourClass) {
            List<Method> result = new ArrayList<Method>();

            for (Method method : behaviourClass.getDeclaredMethods()) {
                if (method.isAnnotationPresent(SPOSHAction.class))
                    result.add(method);
            }
            return result;
        }

        private List<Method> getSenses(Class behaviourClass) {
            List<Method> result = new ArrayList<Method>();

            for (Method method : behaviourClass.getDeclaredMethods()) {
                if (method.isAnnotationPresent(SPOSHSense.class))
                    result.add(method);
            }
            return result;
        }

        /**
         * @return the actions
         */
        public List<Method> getActions() {
            return actions;
        }

        /**
         * @return the senses
         */
        public List<Method> getSenses() {
            return senses;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("Actions:");
            for (Method action : getActions()) {
                sb.append(" ");
                sb.append(action.getName());
            }
            sb.append("; Senses");
            for (Method sense : getSenses()) {
                sb.append(" ");
                sb.append(sense.getName());
            }

            return sb.append(";").toString();
        }
    }
}
