package cz.cuni.amis.pogamut.ut2004.examples.navigationbot;

import java.util.Collection;
import java.util.logging.Level;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathExecutorState;
import cz.cuni.amis.pogamut.base.agent.navigation.IPathFuture;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.math.DistanceUtils;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.ut2004.agent.module.utils.TabooSet;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004PositionHistoryStuckDetector;
import cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector.UT2004TimeStuckDetector;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotModuleController;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.BotKilled;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.ConfigChange;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.InitedMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.utils.UT2004BotRunner;
import cz.cuni.amis.utils.collections.MyCollections;
import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.flag.FlagListener;

/**
 * Example of Simple Pogamut bot, that randomly walks around the map. Bot is uncapable 
 * of handling movers so far.
 *
 * @author Rudolf Kadlec aka ik
 * @author Jimmy
 */
@AgentScoped
public class NavigationBot extends UT2004BotModuleController {    

	protected TabooSet<NavPoint> tabooNavPoints;
	
	protected NavPoint targetNavPoint;
	
    /**
     * Here we can modify initializing command for our bot.
     *
     * @return
     */
    @Override
    public Initialize getInitializeCommand() {
        return new Initialize().setName("NavigationBot");
    }

    /**
     * The bot is initilized in the environment - a physical representation of the
     * bot is present in the game.
     *
     * @param config information about configuration
     * @param init information about configuration
     */
    @SuppressWarnings("unchecked")
	@Override
    public void botInitialized(GameInfo gameInfo, ConfigChange config, InitedMessage init) {
    	// initialize taboo set where we store temporarily unavailable navpoints
    	tabooNavPoints = new TabooSet<NavPoint>(bot);
    	
    	// add stuck detector that watch over the path-following, if it (heuristicly) finds out that the bot has stuck somewhere,
    	// it reports an appropriate path event and the path executor will stop following the path which in turn allows 
    	// us to issue another follow-path command in the right time
        pathExecutor.addStuckDetector(new UT2004TimeStuckDetector(bot, 3.0, 10.0));       // if the bot does not move for 3 seconds, considered that it is stuck
        pathExecutor.addStuckDetector(new UT2004PositionHistoryStuckDetector(bot)); // watch over the position history of the bot, if the bot does not move sufficiently enough, consider that it is stuck
        
        // IMPORTANT
        // adds a listener to the path executor for its state changes, it will allow you to 
        // react on stuff like "PATH TARGET REACHED" or "BOT STUCK"
        pathExecutor.getState().addStrongListener(new FlagListener<IPathExecutorState>() {
			@Override
			public void flagChanged(IPathExecutorState changedValue) {
				switch(changedValue.getState()) {
				case PATH_COMPUTATION_FAILED:
					// if path computation fails to whatever reason, just try another navpoint
				case TARGET_REACHED:
                    // most of the time the execution will go this way
                    goToRandomNavPoint();
                    break;                    
                
                case STUCK:
                	// the bot has stuck! ... target nav point is unavailable currently
                	tabooNavPoints.add(targetNavPoint, 10);
                    // sometimes the bot gets stucked then we have to return him
                    // to some safe location, try more complex maps to get into
                    // this branch
                	
                	// so if we get stuck, we will try to return to some near navpoint                	
                    targetNavPoint = pickSecondNearestNavPoint();
                    
                    if (targetNavPoint == null) { // no more navpoints, all are tabooized
                    	targetNavPoint = pickRandomNavPoint(); // choose one randomly
                    }
                    
                    // let's get to the chosen navpoint
                    pathExecutor.followPath(pathPlanner.computePath(info.getLocation(), targetNavPoint));
                    break;
				}
			}
		});
    }
    
    /**
     * The bot is initilized in the environment - a physical representation of the
     * bot is present in the game.
     *
     * @param config information about configuration
     * @param init information about configuration
     */
    @Override
    public void botSpawned(GameInfo gameInfo, ConfigChange config, InitedMessage init, Self self) {
    	// start the movement
        goToRandomNavPoint();
        // receive logs from the path executor so you can get a grasp on how it is working
        pathExecutor.getLog().setLevel(Level.ALL);
    }
    
    /**
     * Main method that controls the bot - makes decisions what to do next.
     * It is called iteratively by Pogamut engine every time a synchronous batch
     * from the environment is received. This is usually 4 times per second - it
     * is affected by visionTime variable, that can be adjusted in GameBots ini file in
     * UT2004/System folder.
     *
     * @throws cz.cuni.amis.pogamut.base.exceptions.PogamutException
     */
    @Override
    public void logic() throws PogamutException {
    	// mark that another logic iteration has began
    	log.info("--- Logic iteration ---");
    	// log how many navpoints & items the bot knows about and which is visible
    	log.info("Visible navpoints: " + world.getAllVisible(NavPoint.class).size() + " / " + world.getAll(NavPoint.class).size());
    	log.info("Visible items:     " + world.getAllVisible(Item.class).size()     + " / " + world.getAll(Item.class).size());

    	// You might wander why there is no logic here!
    	// Well that's because the bot's logic is completely event driven, check out the botInitialized method
    	// where we're adding a listener to path executor events. When the bot gets stuck or path following is finished,
    	// we're immediately issuing new commands for the bot.
    	//
    	// Note that this might not be the best way to program your bot, but is one of the option you have.
    }

    /**
     * Called each time our bot die. Good for reseting all bot state dependent variables.
     *
     * @param event
     */
    @Override
    public void botKilled(BotKilled event) {
        pathExecutor.stop();
    }

    /**
     * Picks random navpoint and goes to it.
     */
    protected void goToRandomNavPoint() {
        targetNavPoint = pickRandomNavPoint();
        // find path to the random navpoint, path is computed asynchronously
        // so the handle will hold the result onlt after some time
        IPathFuture<ILocated> pathHandle = pathPlanner.computePath(info.getLocation(), targetNavPoint);
        // make the path executor follow the path, executor listens for the
        // asynchronous result of path planning
        pathExecutor.followPath(pathHandle);
    }
    
    /**
     * Randomly picks some navigation point to head to.
     * @return randomly choosed navpoint
     */
    protected NavPoint pickRandomNavPoint() {
        getLog().severe("Picking new target navpoint.");
        
        // 1. get all known navigation points
        Collection<NavPoint> navPoints = tabooNavPoints.filter(getWorldView().getAll(NavPoint.class).values());

        // 2. compute index of the target nav point
        NavPoint chosen = MyCollections.getRandom(navPoints);
        
        if (chosen != null) return chosen;
        
        log.warning("All navpoints are tabooized at this moment, choosing navpoint randomly!");
        
        // all navpoints are tabooized, choose randomly
        return MyCollections.getRandom(getWorldView().getAll(NavPoint.class).values());          
    }

    /**
     * Returns second nearest navpoint that is not inside a taboo list.
     * @return
     */
    protected NavPoint pickSecondNearestNavPoint() {
    	return 
    		DistanceUtils.getSecondNearest(
                tabooNavPoints.filter(getWorldView().getAll(NavPoint.class).values()), // choose second nearest navpoint from this list
                info.getLocation() // "nearest" should be meassured with respect to the current bot's location
            );
    }
    
    public static void main(String args[]) throws PogamutException {
    	// wrapped logic for bots executions, suitable to run single bot in single JVM
    	new UT2004BotRunner(NavigationBot.class, "NavigationBot").setMain(true).startAgent();
    }
}
