package cz.cuni.amis.pogamut.ut2004.examples.botobserver;

import java.util.logging.Level;

import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.AnnotationListenerRegistrator;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.ObjectClassEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base.component.bus.IComponentBus;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.ut2004.agent.params.UT2004AgentParameters;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.ConfigurationObserver;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Initialize;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.InitializeObserver;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.communication.worldview.UT2004WorldView;
import cz.cuni.amis.pogamut.ut2004.factory.guice.remoteagent.UT2004ObserverFactory;
import cz.cuni.amis.pogamut.ut2004.observer.impl.UT2004Observer;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * This simple example will show you how to use Observers.
 * 
 * Observers are very nice as that can collect information about arbitrary bots running in UT2004 game.
 * E.g. you can sense on SELF message of the bot, what it can see, whether it is getting damaged, etc...
 * 
 * @author Jakub Gemrot aka Jimmy
 */
@AgentScoped
public class BotObserver extends UT2004Observer {

	@Inject
	public BotObserver(UT2004AgentParameters params, IComponentBus bus, IAgentLogger agentLogger, UT2004WorldView worldView, IAct act) {
		super(params, bus, agentLogger, worldView, act);
		if (!(params instanceof BotObserverParams)) throw new RuntimeException("BotObserver must be instantiated with BotObserverParams not " + params.getClass().getSimpleName() + ".");
		getLogger().addDefaultConsoleHandler();
		listenerRegistrator = new AnnotationListenerRegistrator(this, getWorldView(), getLogger().getCategory("AnnotatedListeners"));
		listenerRegistrator.getLog().setLevel(Level.ALL);
		listenerRegistrator.addListeners();
		listenerRegistrator.getLog().setLevel(Level.WARNING);
	}

	/**
	 * This is here in order for Object/EventListener annotations to work.
	 */
	private AnnotationListenerRegistrator listenerRegistrator;
	
	/**
	 * Whether we're currently observing some bot or not.
	 */
	private boolean observing = false;
	
	/**
	 * We're going to instantiate {@link BotObserver} only with {@link BotObserverParams} (see constructor that enforces that).
	 */
	public BotObserverParams getParams() {
		return (BotObserverParams)super.getParams();
	};
	
	@ObjectClassEventListener(eventClass=WorldObjectUpdatedEvent.class, objectClass=Player.class)
	public void playerEvent(WorldObjectUpdatedEvent<Player> event) {
		if (!observing) {
			if (event.getObject().getId().getStringId().contains(getParams().getBotNameToObserve()) ||
				event.getObject().getName().contains(getParams().getBotNameToObserve())	
			) {
				// TELL WE WANT TO OBSERVE THE PLAYER
				getAct().act(new InitializeObserver().setId(event.getObject().getId().getStringId()));
				// CONFIGURE WHICH MESSAGES WE WANT TO RECEIVE
				getAct().act(
					new ConfigurationObserver()
						.setAll(true)      // WE WANT TO RECEIVE UPDATES	
						.setUpdate(0.25)   // UPDATE TIME						
						.setSelf(true)     // WE WANT TO RECEIVE "Self"
						.setAsync(true)    // export Async events (BotDamaged, PlayerDamaged, BotKilled, PlayerKilled, HearNoise, etc.)
						.setGame(false)    // Export GAME message
						.setSee(false)	   // NavPoints, Items, Players
						.setSpecial(false) // SPECIFIC GAME TYPE MESSAGES (BOM, FLG, DOM)
				);
				observing = true;
	            log.info("START OBSERVING: " + event.getObject().getName());
	            log.info("START OBSERVING: " + event.getObject());
			} else {
				log.info("NOT OBSERVING, NAME MISMATCH: " + event.getObject().getName());
				log.info("NOT OBSERVING, NAME MISMATCH: " + event.getObject());
			}
		}
	}
	
	@ObjectClassEventListener(eventClass=WorldObjectUpdatedEvent.class, objectClass=Self.class)
	public void selfEvent(WorldObjectUpdatedEvent<Self> event) {
		Self self = event.getObject();
		log.info("Sensed: " + self);
	}

    /**
     * In case you starting this observer manually, use this method.
     * 
     * Just configures the NAME of the bot to observe.
     * 
     * NOTE THAT THE OBSERVER MUST BE STARTED AFTER THE OBSERVED BOT!
     * 
     * @param args
     */
	public static void startObserver() {
		BotObserverParams params = new BotObserverParams();
        params.setAgentId(new AgentId("Observer"));
        params.setWorldAddress(new SocketConnectionAddress("127.0.0.1", 3002));
        
        // PUT HERE NAME OF THE BOT WE WANT TO OBSERVE
        params.setBotNameToObserve("ObservedBot");
        
        // Creating module that will tell Guice to instantiate our Observer
        ObserverConnectionModule module = new ObserverConnectionModule();
        // Creating Observer factory 
        UT2004ObserverFactory observerFactory = new UT2004ObserverFactory(module);
        BotObserver observer = (BotObserver) observerFactory.newAgent(params);

        // Start our observer
        observer.start();
	}

    public static void main(String args[]) throws PogamutException {
        startObserver();        
    }
}
