/**
 * 
 */
package sk.stuba.fiit.pogamut.jungigation.objects;

import java.io.BufferedWriter;
import java.io.CharArrayWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import jung.myio.GraphMLWriter2;
import jung.myio.MetadataEdgeTransformer;
import jung.myio.MetadataGraphTransformer;
import jung.myio.MetadataNodeTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.EdgeMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.GraphMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.HyperEdgeMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.NodeMetadataTransformer;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointNeighbourLink;
import edu.uci.ics.jung.graph.util.EdgeType;
import edu.uci.ics.jung.io.GraphIOException;
import edu.uci.ics.jung.io.graphml.GraphMLReader2;

/**
 * <p>
 * Class for providing navigation graphs for advanced navigation in game.
 * </p>
 * <p>
 * To access to some low level methods you can inherit from this class and access its protected methods.
 * </p>
 * 
 * @author LuVar
 * 
 */
public class NavigationGraphProviderForMap {
	private static NavigationGraphProviderForMap singleton = null;
	private static final Logger log = Logger.getLogger("NavigationGraphProviderForMap");
	private final String directoryWithGMLFiles;

	private final Map<String, NavigationGraphSynchronized> cache = Collections.synchronizedMap(new HashMap<String, NavigationGraphSynchronized>());

	/**
	 * <p>
	 * Class responsible for loading *.graphML files with navigation informations stored for particular map.
	 * </p>
	 * <p>
	 * Note that you can use this class also as singleton in your bots, if there will be running more than one at once.
	 * </p>
	 * 
	 * @param directoryWithGMLFiles path, where to look for *.graphML files for maps
	 * 
	 * @see #getInstance()
	 */
	public NavigationGraphProviderForMap(String directoryWithGMLFiles) {
		this.directoryWithGMLFiles = new File(directoryWithGMLFiles).getPath() + File.separator;
	}

	/**
	 * <p>
	 * </p>
	 * 
	 * @return instance of {@link NavigationGraphProviderForMap} which is looking for gml files in actual directory
	 */
	public static NavigationGraphProviderForMap getInstance() {
		if (NavigationGraphProviderForMap.singleton == null) {
			NavigationGraphProviderForMap.singleton = new NavigationGraphProviderForMap("./");
		}
		return NavigationGraphProviderForMap.singleton;
	}

	/**
	 * <p>
	 * Method tries to load map data from file named like map name with gml suffix. For example "CTF-Lostfaith.gml".
	 * Second parameter is collection of {@link NavPoint}-s obtained from game by bot. It is for checking, if saved file
	 * corresponds with actual map. If it does not correspond (same vertices in booth datasets), than navigation graph
	 * is constructed from given navpoints.
	 * </p>
	 * <p>
	 * To check if {@link NavPoint}-s is representing same graph as stored in gml file, use
	 * {@link #checkCompatibility()} method.
	 * </p>
	 * <p>
	 * For more calls with same map name parameter returns same synchronized navigation graph as for all calls (more
	 * bots share same instance).
	 * </p>
	 * 
	 * @param mapName name of map, for example "CTF-Maul". Can be obtained from bot by
	 *            <code>mapName = this.gameInfo.getLevel();</code>.
	 * @param navpoints can be obtained by
	 *            <code>navpoints = this.bot.getWorldView().getAll(NavPoint.class).values();</code>.
	 * @return
	 */
	public NavigationGraphSynchronized getNavigationDataForMapWithChceckAndInitialization(String mapName, Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g = this.cache.get(mapName);
		if (g == null) {
			g = NavigationGraphProviderForMap.getGraphFromNavPoints(navpoints);
			g = NavigationGraphProviderForMap.getGraphFromFileAndCompare(this.mapNameToFilename(mapName), g);
			this.cache.put(mapName, g);
		}
		return g;
	}

	public NavigationGraphSynchronized getNavigationDataForMap(String mapName) {
		NavigationGraphSynchronized g = this.cache.get(mapName);
		if (g == null) {
			try {
				g = NavigationGraphProviderForMap.getGraphFromFile(this.mapNameToFilename(mapName));
			} catch (Exception ex) {
				throw new RuntimeException("Map graph not found in cache nor on disk! Use getNavigationDataForMapWithChceckAndInitialization method first!");
			}
			if (g == null) {
				throw new RuntimeException("Map graph not found in cache nor on disk! Use getNavigationDataForMapWithChceckAndInitialization method first!");
			}
		}
		return g;
	}

	/**
	 * <p>
	 * Returns map only from cache!!! TODO try to read from file.
	 * </p>
	 * 
	 * @param mapName
	 * @return
	 */
	private String mapNameToFilename(String mapName) {
		return this.directoryWithGMLFiles + mapName + ".graphML";
	}

	/**
	 * <p>
	 * Reads navigation graph from file and compares it with given navpoints. If file doesnot exist, returns false.
	 * </p>
	 * 
	 * @param mapName name of map. This is used to construct filename and open file from disk
	 * @param navpoints navpoints from actual game
	 * @return true, if navigation graph stored on disk is compatible with given navpoints
	 */
	public boolean checkCompatibility(String mapName, Collection<NavPoint> navpoints) {
		return checkCompatibilityWithFile(this.mapNameToFilename(mapName), navpoints);
	}

	protected static boolean checkCompatibilityWithFile(String filename, Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g = getGraphFromNavPoints(navpoints);
		NavigationGraphSynchronized gF = getGraphFromFile(filename);
		// TODO edges control missing. Control which vertices is connected to which.
		return ((gF.getVertices().containsAll(g.getVertices())) && (g.getVertices().containsAll(gF.getVertices())));
	}

	/**
	 * <p>
	 * Tries to open graph file from disc. If it is successful, it will compare vertices of graph in parameter g with it
	 * and if they are same, it will return loaded graph from file. It will in any other case return given graph
	 * reference.
	 * </p>
	 * 
	 * @param filename path to file, which should be read
	 * @param g graph to compare with
	 * @return graph from file, if read graph has same vertices. Given graph in other case
	 */
	protected static NavigationGraphSynchronized getGraphFromFileAndCompare(String filename, NavigationGraphSynchronized g) {
		NavigationGraphSynchronized gF;
		try {
			gF = NavigationGraphProviderForMap.getGraphFromFile(filename);
		} catch (Exception ex) {
			log.info("Could not load file " + filename + "! Using navigation graph from game. Error: " + ex.getMessage());
			return g;
		}
		Collection<MyVertice> gFvertices = gF.getVertices();
		Collection<MyVertice> gvertices = g.getVertices();
		if ((gFvertices.containsAll(gvertices)) && (gvertices.containsAll(gFvertices))) {
			// TODO edges control missing. Control which vertices is connected to which.
			log.info("Saved graph in file was same as generated from game.");
			return gF;
		}
		List<MyVertice> gfv = new LinkedList<MyVertice>(gFvertices);
		Collections.sort(gfv);
		List<MyVertice> gv = new LinkedList<MyVertice>(gvertices);
		Collections.sort(gv);
		System.out.println("FILE VERTICES:");
		for (MyVertice myVertice : gfv) {
			System.out.println(myVertice);
		}
		System.out.println("GAME VERTICES:");
		for (MyVertice myVertice : gv) {
			System.out.println(myVertice);
		}
		log.severe("Saved graph was not same as graph generated from game!");
		return g;
	}// end of method getGraphFromFileAndCompare

	/**
	 * <p>
	 * Constructs navigation graph from given {@link NavPoint} collection.
	 * </p>
	 * 
	 * @param navpoints {@link NavPoint}-s collection
	 * @return graph constructed from given collection
	 */
	protected static NavigationGraphSynchronized getGraphFromNavPoints(Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g;
		g = new NavigationGraphSynchronized(0.0, 0, 0);
		for (NavPoint navPoint : navpoints) {
			MyVertice from = new MyVertice(navPoint);
			g.addVertex(from);
			Collection<NavPointNeighbourLink> neighobur = navPoint.getOutgoingEdges().values();
			for (NavPointNeighbourLink navPointNeighbourLink : neighobur) {
				MyVertice to = new MyVertice(navPointNeighbourLink.getToNavPoint());
				g.addEdge(new MyEdge(from.getId().getStringId() + "->" + to.getId().getStringId()), from, to, EdgeType.DIRECTED);
			}
		}
		return g;
	}

	/**
	 * <p>
	 * Reads graph from given file.
	 * </p>
	 * 
	 * @param filename filename of graph navigation gml file
	 * @return
	 * 
	 * @throws RuntimeException if problem occurred in opening or reading graph file
	 */
	public static NavigationGraphSynchronized getGraphFromFile(String filename) {
		NavigationGraphSynchronized graphFromFile;
		GraphMLReader2<NavigationGraphSynchronized, MyVertice, MyEdge> gr;
		try {
			gr = new GraphMLReader2<NavigationGraphSynchronized, MyVertice, MyEdge>(new FileReader(filename), new GraphMetadataTransformer(), new NodeMetadataTransformer(), new EdgeMetadataTransformer(), new HyperEdgeMetadataTransformer());
		} catch (FileNotFoundException ex) {
			throw new RuntimeException("File not found! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		} catch (Exception ex) {
			throw new RuntimeException("Exception while opening graph file. file=\"" + filename + "\"! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		}
		try {
			graphFromFile = gr.readGraph();
		} catch (GraphIOException ex) {
			throw new RuntimeException("GraphIOException while reading graph! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		} catch (Exception ex) {
			throw new RuntimeException("Exception while reading graph from file. file=\"" + filename + "\"! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		}
		return graphFromFile;
	}// end of method getGraphFromFile

	public void startLearningMap(final String mapName) {
		if (this.cache.get(mapName) == null) {
			throw new RuntimeException("Cant start saving map, if map is not initialized. Use getNavigationDataForMapWithChceckAndInitialization method first!");
		}
		NavigationGraphProviderForMap.startAutosavingMap(this.mapNameToFilename(mapName), this.cache.get(mapName));
	}

	/**
	 * <p>
	 * Method will try to find given graph instance in local {@link #cache} map. If found, it will use map name from
	 * {@link #cache} map and save file to filesystem. If no navgraph is not found, exception will be thrown.
	 * </p>
	 * 
	 * @param navgraph
	 * @throws IOException
	 */
	public void saveNavigationGraph(NavigationGraphSynchronized navgraph) throws IOException {
		for (Entry<String, NavigationGraphSynchronized> entry : new ArrayList<Entry<String, NavigationGraphSynchronized>>(this.cache.entrySet())) {
			if (entry.getValue() != null && entry.getValue().equals(navgraph)) {
				NavigationGraphProviderForMap.saveGraphToFile(this.mapNameToFilename(entry.getKey()), navgraph);
				return;
			}
		}// end of foreach entry in new ArrayList<Entry<String, NavigationGraphSynchronized>>(this.cache.values())
		throw new IllegalArgumentException("Given navigation graph was not found in local cache map! Use appropriete method to get nav graph!");
	}

	/**
	 * <p>
	 * Starts and tries to save selected map every one minute. Useful when learning some map and adding data to graph in
	 * any way.
	 * </p>
	 * <p>
	 * TODO no management, no stopping, or checking if more threads is saving same graph!
	 * </p>
	 * 
	 * @param filename
	 */
	protected static void startAutosavingMap(final String filename, final NavigationGraphSynchronized g) {
		try {
			saveGraphToFile(filename, g);
		} catch (Exception ex) {
			log.severe("Error while first autosaving! I give up and end! Error:" + ex.getMessage() + ",stack:" + ex.getStackTrace());
			throw new RuntimeException("Error while first autosaving! I give up and end! Error:" + ex.getMessage(), ex);
		}
		Thread vlakno = new Thread(new Runnable() {
			@Override
			public void run() {
				while (true) {
					try {
						Thread.sleep(60 * 1000);
					} catch (InterruptedException ex) {
						log.info("End of saving file " + filename);
						break;
					}
					try {
						saveGraphToFile(filename, g);
					} catch (Exception ex) {
						log.severe("Error while autosaving in thread! Error:" + ex.getMessage() + ",stack:" + ex.getStackTrace());
					}
				}
			}
		});
		vlakno.setName("Saving thread for file " + filename);
		vlakno.setDaemon(true);
		vlakno.start();
	}

	protected static void saveGraphToFile(final String filename, final NavigationGraphSynchronized g) throws IOException {
		long timeStart = System.currentTimeMillis();
		/*
		 * GraphMLWriter gw = new GraphMLWriter<MyVertice, MyEdge>(); //TODO nejako doriesit serializaciu a
		 * deserializaciu!!! gw.setVertexDescriptions(new SerializableToStringTransformer()); gw.setVertexIDs(new
		 * SerializableToStringTransformer()); gw.setEdgeIDs(new SerializableToStringTransformer()); try {
		 * log.finest("Saving gml file."); gw.save(g, new FileWriter(filename)); } catch (IOException ex) {
		 * Logger.getLogger("AbstractBotWithJungigation").log(Level.SEVERE, null, ex); }
		 */
		CharArrayWriter aw = new CharArrayWriter(900000);
		GraphMLWriter2<NavigationGraphSynchronized, MyVertice, MyEdge> gw;
		gw = new GraphMLWriter2<NavigationGraphSynchronized, MyVertice, MyEdge>(aw, new MetadataGraphTransformer(), new MetadataNodeTransformer(), new MetadataEdgeTransformer(), null);
		long filewritingTimeStart = 0;
		try {
			log.finest("Saving graphML file.");
			gw.writeGraph(g);
			// Should avoid of incomplete files.
			filewritingTimeStart = System.currentTimeMillis();
			FileWriter fw = new FileWriter(filename);
			BufferedWriter bw = new BufferedWriter(fw);
			bw.write(aw.toCharArray());
			bw.flush();
		} catch (GraphIOException ex) {
			Logger.getLogger("AbstractBotWithJungigation").log(Level.SEVERE, null, ex);
		}
		long timeEnd = System.currentTimeMillis();
		log.info("Gml saved to file in " + (timeEnd - timeStart) + " ms (writing to disk was " + (timeEnd - filewritingTimeStart) + "ms).");
	}
}
