package sk.stuba.fiit.pogamut.jungigation.worldInfo.prophets;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import sk.stuba.fiit.pogamut.jungigation.objects.NavigationGraphProviderForMap;
import sk.stuba.fiit.pogamut.jungigation.objects.NavigationGraphSynchronized;
import sk.stuba.fiit.pogamut.jungigation.worldInfo.GameTime;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.Items;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.worldview.map.Box;

/**
 * <p>
 * Class responsible for managing informations about known world. It can provide informations
 * for bots in DM and also in team (CTF) game modes. For DM modes make for every bot a single
 * instance by normal constructor. For team games use {@link #getInstance(IVisionWorldView, int)
 * method. 
 * </p>
 * <p>
 * WorldProphet should implements all smaller prophets method. Create interfaces and implement them...
 * </p>
 * 
 * @author LuVar
 * 
 * @see	Items
 */
public class WorldProphet {
	private static final Map<Integer, WorldProphet> mapForTeamGame = Collections.synchronizedMap(new HashMap<Integer, WorldProphet>());
	
	private final GamersProphet gamersProphet;
	private final ItemsProphet itemsProphet;
	private final FlagsProphet flagsProphet;
	
	private final int team;
	private final GameInfo gameInfo;
	private final Box mapBox;
	private final WorldProphetFrame frame;
	private final NavigationGraphSynchronized navigationMap;
	
	/**
	 * <p>
	 * This constructor should be used only in DM and another non-team game plays. It will provide instance of
	 * {@link WorldProphet} class for one player vision.
	 * </p>
	 * <p>
	 * For team types of games use {@link #getInstance(IVisionWorldView, int)} method. It will obtain unite
	 * {@link WorldProphet} view to the world. Don't forget to call that method from all team bots in one team.
	 * </p>
	 * 
	 * @param worldView
	 * @param gameInfo
	 */
	public WorldProphet(IWorldView worldView, GameInfo gameInfo) {
		this(worldView, AgentInfo.TEAM_NONE, gameInfo, true);
	}
	
	/**
	 * 
	 * @param worldView
	 * @param gameInfo
	 * @param shouldBeProphetWindowOpenned
	 */
	public WorldProphet(IWorldView worldView, GameInfo gameInfo, boolean shouldBeProphetWindowOpenned) {
		this(worldView, AgentInfo.TEAM_NONE, gameInfo, shouldBeProphetWindowOpenned);
	}
	
	/**
	 * <p>
	 * For internal use only.
	 * </p>
	 * 
	 * @param worldView
	 * @param team specifies team for which this {@link WorldProphet} is created
	 */
	private WorldProphet(IWorldView worldView, int team, GameInfo gameInfo, boolean shouldBeProphetWindowOpenned) {
		this.team = team;
		this.gameInfo = gameInfo;
		
		this.gamersProphet = new GamersProphet(worldView, this.team);
		this.itemsProphet = new ItemsProphet(worldView, this.gamersProphet);
		this.flagsProphet = new FlagsProphet(worldView, this.team, this.gameInfo, this.gamersProphet);
		
		this.navigationMap = NavigationGraphProviderForMap.getInstance().getNavigationDataForMap(this.gameInfo.getLevel());
		
		Collection<NavPoint> navpoints = worldView.getAll(NavPoint.class).values();
		this.mapBox = WorldProphet.calculateBox(navpoints);
		
		if (shouldBeProphetWindowOpenned) {
			this.frame = new WorldProphetFrame();
			this.frame.getWorldProphetPanel().setWorldProphet(this);
			this.frame.setVisible(true);
		} else {
			this.frame = null;
		}
	}
	
	/**
	 * TODO should be deleted and here should be all methods from it
	 * 
	 * @return
	 */
	public FlagsProphet getFlagsProphet() {
		return this.flagsProphet;
	}
	
	/**
	 * TODO should be deleted and here should be all methods from it
	 * 
	 * @return
	 */
	public GamersProphet getGamersProphet() {
		return this.gamersProphet;
	}
	
	/**
	 * TODO should be deleted and here should be all methods from it
	 * 
	 * @return
	 */
	public ItemsProphet getItemsProphet() {
		return this.itemsProphet;
	}
	
	private static Box calculateBox(Collection<NavPoint> navpoints) {
		double minX = Double.MAX_VALUE, maxX = Double.MIN_VALUE;
		double minY = Double.MAX_VALUE, maxY = Double.MIN_VALUE;
		double minZ = Double.MAX_VALUE, maxZ = Double.MIN_VALUE;
		for (NavPoint navPoint : navpoints) {
			double x = navPoint.getLocation().x;
			if (minX > x) {
				minX = x;
			} else {
				if (maxX < x) {
					maxX = x;
				}
			}// end of if else minX > x
			double y = navPoint.getLocation().y;
			if (minY > y) {
				minY = y;
			} else {
				if (maxY < y) {
					maxY = y;
				}
			}// end of if else minX > x
			double z = navPoint.getLocation().z;
			if (minZ > z) {
				minZ = z;
			} else {
				if (maxZ < z) {
					maxZ = z;
				}
			}// end of if else minX > x
		}// end of foreach navpoint in navpoints
		return new Box(minX, minY, minZ, maxX, maxY, maxZ);
	}
	
	/**
	 * <p>
	 * Gets "singleton" instance of {@link WorldProphet} class. There is one instance of {@link WorldProphet} per team.
	 * </p>
	 * <p>
	 * Don't forget to call that method from all team bots in one team. This methods also registers bots
	 * {@link IVisionWorldView} to internal structure.
	 * </p>
	 * 
	 * @param worldView
	 * @param team
	 * @return
	 */
	public static WorldProphet getInstance(IWorldView worldView, int team, GameInfo info, boolean shouldBeProphetWindowOpenned) {
		WorldProphet navrat = WorldProphet.mapForTeamGame.get(team);
		if (navrat == null) {
			GameTime.forceStartUpdatingTime(worldView);
			navrat = new WorldProphet(worldView, team, info, shouldBeProphetWindowOpenned);
			WorldProphet.mapForTeamGame.put(team, navrat);
		}
		navrat.addMoreWorldView(worldView);
		return navrat;
	}
	
	public static WorldProphet getInstance(IWorldView worldView, int team, GameInfo info) {
		return WorldProphet.getInstance(worldView, team, info, false);
	}
	
	protected void addMoreWorldView(IWorldView worldView) {
		this.itemsProphet.addAnotherViewer(worldView);
		this.gamersProphet.addAnotherViewer(worldView);
		this.flagsProphet.addAnotherViewer(worldView);
	}
	
	public void removeViewer(IWorldView worldView) {
		this.itemsProphet.removeViewer(worldView);
		this.gamersProphet.removeViewer(worldView);
		this.flagsProphet.removeViewer(worldView);
	}
	
	public Box getMapBox() {
		return this.mapBox;
	}
	
	public int getTeam() {
		return this.team;
	}
}
