package sk.stuba.fiit.pogamut.jungigation.transformers;

import java.util.Map.Entry;

import jung.myio.GraphMLWriter2;

import org.apache.commons.collections15.Transformer;
import org.apache.log4j.Logger;

import sk.stuba.fiit.pogamut.jungigation.objects.NavigationGraphSynchronized;
import edu.uci.ics.jung.io.graphml.GraphMetadata;
import edu.uci.ics.jung.io.graphml.GraphMetadata.EdgeDefault;

/**
 * <p>
 * Class used for "serialization" of navigation graph. Constructor of {@link GraphMLWriter2} class takes this
 * class as transformer from {@link NavigationGraphSynchronized} to {@link GraphMetadata}.
 * </p>
 * <p>
 * This transformer transforms few attributes from {@link NavigationGraphSynchronized} instance
 * ({@link NavigationGraphSynchronized#PROPERTY_LearningTime},
 * {@link NavigationGraphSynchronized#PROPERTY_RespawnCount} and
 * {@link NavigationGraphSynchronized#PROPERTY_StucksCount}) and than tryies to serialize also all
 * additional attributes which can be obtained from {@link NavigationGraphSynchronized#getAdditionalProperties()}
 * method call. 
 * </p>
 * 
 * @author LuVar
 *
 */
public class MetadataGraphTransformer implements Transformer<NavigationGraphSynchronized, GraphMetadata> {
    private static final Logger log = Logger.getLogger(MetadataGraphTransformer.class);

    /**
     * <p>
     * Transforms {@link NavigationGraphSynchronized} instances attributes into {@link GraphMetadata}. Transformed
     * object is mainly used for serialization to file. More informations can be found in class description
     * (here: {@link MetadataGraphTransformer}).
     * </p>
     */
    @Override
    public GraphMetadata transform(NavigationGraphSynchronized graph) {
	GraphMetadata gm = new GraphMetadata();
	switch (graph.getDefaultEdgeType()) {
	case DIRECTED:
	    gm.setEdgeDefault(EdgeDefault.DIRECTED);
	    break;
	case UNDIRECTED:
	    gm.setEdgeDefault(EdgeDefault.UNDIRECTED);
	    break;
	default:
	    throw new RuntimeException("Graph method getDefaultEdgeType() returned unknown value! Value=" + graph.getDefaultEdgeType() + ".");
	}
	gm.setGraph(graph);

	try {
	    gm.setProperty(NavigationGraphSynchronized.PROPERTY_LearningTime, Double.toString(graph.getLearningTime()));
	    gm.setProperty(NavigationGraphSynchronized.PROPERTY_StucksCount, Long.toString(graph.getStucksCount()));
	    gm.setProperty(NavigationGraphSynchronized.PROPERTY_RespawnCount, Long.toString(graph.getRespawnCount()));
	} catch (Exception ex) {
	    log.error("Error while setting some property! Error:" + ex.getMessage(), ex);
	}

	for (Entry<String, String> entry : graph.getAdditionalProperties().entrySet()) {
	    try {
		gm.setProperty(entry.getKey(), entry.getValue());
	    } catch (Exception ex) {
		log.error("Error while setting additional property (" + " - " + "). Error:" + ex.getMessage(), ex);
	    }
	}
	return gm;
    }
}
