/**
 * 
 */
package sk.stuba.fiit.pogamut.jungigation.objects;

import java.io.BufferedWriter;
import java.io.CharArrayWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jung.myio.GraphMLWriter2;

import org.apache.log4j.Logger;

import sk.stuba.fiit.pogamut.jungigation.transformers.EdgeMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.GraphMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.HyperEdgeMetadataTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.MetadataEdgeTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.MetadataGraphTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.MetadataNodeTransformer;
import sk.stuba.fiit.pogamut.jungigation.transformers.NodeMetadataTransformer;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointNeighbourLink;
import edu.uci.ics.jung.graph.util.EdgeType;
import edu.uci.ics.jung.io.GraphIOException;
import edu.uci.ics.jung.io.graphml.GraphMLReader2;

/**
 * <p>
 * Class for providing navigation graphs for advanced navigation in game.
 * </p>
 * 
 * @author LuVar
 * 
 */
public class NavigationGraphProviderForMap {
	private static NavigationGraphProviderForMap singleton = null;
	private static final Logger log = Logger.getLogger(NavigationGraphSynchronized.class);
	private final String directoryWithGMLFiles;

	private final Map<String, NavigationGraphSynchronized> cache = Collections.synchronizedMap(new HashMap<String, NavigationGraphSynchronized>());

	/**
	 * <p>
	 * Class responsible for loading *.graphML files with navigation informations stored for particular map.
	 * </p>
	 * <p>
	 * Note that you can use this class also as singleton in your bots, if there will be running more than one at once.
	 * </p>
	 * 
	 * @param directoryWithGMLFiles path, where to look for *.graphML files for maps
	 * 
	 * @see #getInstance()
	 */
	public NavigationGraphProviderForMap(String directoryWithGMLFiles) {
		this.directoryWithGMLFiles = new File(directoryWithGMLFiles).getPath() + File.separator;
	}

	/**
	 * <p>
	 * </p>
	 * 
	 * @return instance of {@link NavigationGraphProviderForMap} which is looking for gml files in actual directory
	 */
	public static NavigationGraphProviderForMap getInstance() {
		if (NavigationGraphProviderForMap.singleton == null) {
			NavigationGraphProviderForMap.singleton = new NavigationGraphProviderForMap("./");
		}
		return NavigationGraphProviderForMap.singleton;
	}

	/**
	 * <p>
	 * Method tries to load map data from file named like map name with gml suffix. For example "CTF-Lostfaith.gml".
	 * Second parameter is collection of {@link NavPoint}-s obtained from game by bot. It is for checking, if saved file
	 * corresponds with actual map. If it does not correspond (same vertices in booth datasets), than navigation graph
	 * is constructed from given navpoints.
	 * </p>
	 * <p>
	 * To check if {@link NavPoint}-s is representing same graph as stored in gml file, use
	 * {@link #checkCompatibility()} method.
	 * </p>
	 * <p>
	 * For more calls with same map name parameter returns same synchronized navigation graph as for all calls (more
	 * bots share same instance).
	 * </p>
	 * 
	 * @param mapName name of map, for example "CTF-Maul". Can be obtained from bot by
	 *            <code>mapName = this.gameInfo.getLevel();</code>.
	 * @param navpoints can be obtained by
	 *            <code>navpoints = this.bot.getWorldView().getAll(NavPoint.class).values();</code>.
	 * @return
	 */
	public NavigationGraphSynchronized getNavigationDataForMapWithChceckAndInitialization(String mapName, Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g = this.cache.get(mapName);
		if (g == null) {
			g = NavigationGraphProviderForMap.getGraphFromNavPoints(navpoints);
			g = NavigationGraphProviderForMap.getGraphFromFileAndCompare(this.mapNameToFilename(mapName), g);
			this.cache.put(mapName, g);
		}
		return g;
	}

	public NavigationGraphSynchronized getNavigationDataForMap(String mapName) {
		NavigationGraphSynchronized g = this.cache.get(mapName);
		if (g == null) {
			try {
				g = NavigationGraphProviderForMap.getGraphFromFile(this.mapNameToFilename(mapName));
			} catch (Exception ex) {
				throw new RuntimeException("Map graph not found in cache nor on disk! Use getNavigationDataForMapWithChceckAndInitialization method first!");
			}
			if (g == null) {
				throw new RuntimeException("Map graph not found in cache nor on disk! Use getNavigationDataForMapWithChceckAndInitialization method first!");
			}
		}
		return g;
	}

	/**
	 * <p>
	 * Returns map only from cache!!! TODO try to read from file.
	 * </p>
	 * 
	 * @param mapName
	 * @return
	 */
	private String mapNameToFilename(String mapName) {
		return this.directoryWithGMLFiles + mapName + ".graphML";
	}

	/**
	 * <p>
	 * Reads navigation graph from file and compares it with given navpoints. If file doesnot exist, returns false.
	 * </p>
	 * 
	 * @param mapName name of map. This is used to construct filename and open file from disk
	 * @param navpoints navpoints from actual game
	 * @return true, if navigation graph stored on disk is compatible with given navpoints
	 */
	public boolean checkCompatibility(String mapName, Collection<NavPoint> navpoints) {
		return checkCompatibilityWithFile(this.mapNameToFilename(mapName), navpoints);
	}

	/**
	 * <p>
	 * Method not mean to be used. This method should be private, but can be sometimes useful.
	 * </p>
	 * <p>
	 * Method checks for compatibility between given navpoints set and given graph by filename.
	 * Method will return true if constructed graphs have same vertices. Edges can vary.
	 * </p>
	 * 
	 * @param filename	filename of graphML file for graph
	 * @param navpoints	navigation points from UT2004 game
	 * @return		returns if booth graphs have same vertices
	 */
	public static boolean checkCompatibilityWithFile(String filename, Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g = getGraphFromNavPoints(navpoints);
		NavigationGraphSynchronized gF = getGraphFromFile(filename);
		// TODO edges control missing. Control which vertices is connected to which.
		return ((gF.getVertices().containsAll(g.getVertices())) && (g.getVertices().containsAll(gF.getVertices())));
	}

	/**
	 * <p>
	 * Method not mean to be used. This method should be private, but can be sometimes useful. You should
	 * use {@link #getNavigationDataForMapWithChceckAndInitialization(String, Collection)} method instead.
	 * </p>
	 * <p>
	 * Tries to open graph file from disc. If it is successful, it will compare vertices of graph in parameter g with it
	 * and if they are same, it will return loaded graph from file. It will in any other case return given graph
	 * reference.
	 * </p>
	 * 
	 * @param filename path to file, which should be read
	 * @param g graph to compare with
	 * @return graph from file, if read graph has same vertices. Given graph in other case
	 */
	public static NavigationGraphSynchronized getGraphFromFileAndCompare(String filename, NavigationGraphSynchronized g) {
		NavigationGraphSynchronized gF;
		try {
			gF = NavigationGraphProviderForMap.getGraphFromFile(filename);
		} catch (Exception ex) {
			log.info("Could not load file " + filename + "! Using navigation graph from game. Error: " + ex.getMessage());
			return g;
		}
		Collection<MyVertice> gFvertices = gF.getVertices();
		Collection<MyVertice> gvertices = g.getVertices();
		if ((gFvertices.containsAll(gvertices)) && (gvertices.containsAll(gFvertices))) {
			// TODO edges control missing. Control which vertices is connected to which.
			log.info("Saved graph in file was same as generated from game.");
			return gF;
		}
		List<MyVertice> gfv = new LinkedList<MyVertice>(gFvertices);
		Collections.sort(gfv);
		List<MyVertice> gv = new LinkedList<MyVertice>(gvertices);
		Collections.sort(gv);
		System.out.println("FILE VERTICES:");
		for (MyVertice myVertice : gfv) {
			System.out.println(myVertice);
		}
		System.out.println("GAME VERTICES:");
		for (MyVertice myVertice : gv) {
			System.out.println(myVertice);
		}
		log.error("Saved graph was not same as graph generated from game!");
		return g;
	}// end of method getGraphFromFileAndCompare

	/**
	 * <p>
	 * Method not mean to be used. This method should be private, but can be sometimes useful.
	 * You should use {@link #getNavigationDataForMapWithChceckAndInitialization(String, Collection)}
	 * method instead.
	 * </p>
	 * <p>
	 * Constructs navigation graph from given {@link NavPoint} collection.
	 * </p>
	 * 
	 * @param navpoints {@link NavPoint}-s collection
	 * @return graph constructed from given collection
	 */
	public static NavigationGraphSynchronized getGraphFromNavPoints(Collection<NavPoint> navpoints) {
		NavigationGraphSynchronized g;
		g = new NavigationGraphSynchronized(0.0, 0, 0);
		for (NavPoint navPoint : navpoints) {
			MyVertice from = new MyVertice(navPoint);
			g.addVertex(from);
			Collection<NavPointNeighbourLink> neighobur = navPoint.getOutgoingEdges().values();
			for (NavPointNeighbourLink navPointNeighbourLink : neighobur) {
				MyVertice to = new MyVertice(navPointNeighbourLink.getToNavPoint());
				g.addEdge(new MyEdge(from.getId().getStringId() + "->" + to.getId().getStringId()), from, to, EdgeType.DIRECTED);
			}
		}
		return g;
	}

	/**
	 * <p>
	 * Reads graph from given file.
	 * </p>
	 * 
	 * @param filename filename of graph navigation gml file
	 * @return
	 * 
	 * @throws RuntimeException if problem occurred in opening or reading graph file
	 */
	public static NavigationGraphSynchronized getGraphFromFile(String filename) {
		NavigationGraphSynchronized graphFromFile;
		GraphMLReader2<NavigationGraphSynchronized, MyVertice, MyEdge> gr;
		try {
			gr = new GraphMLReader2<NavigationGraphSynchronized, MyVertice, MyEdge>(new FileReader(filename), new GraphMetadataTransformer(), new NodeMetadataTransformer(), new EdgeMetadataTransformer(), new HyperEdgeMetadataTransformer());
		} catch (FileNotFoundException ex) {
			throw new RuntimeException("File not found! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		} catch (Exception ex) {
			throw new RuntimeException("Exception while opening graph file. file=\"" + filename + "\"! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		}
		try {
			graphFromFile = gr.readGraph();
		} catch (GraphIOException ex) {
			throw new RuntimeException("GraphIOException while reading graph! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		} catch (Exception ex) {
			throw new RuntimeException("Exception while reading graph from file. file=\"" + filename + "\"! file=\"" + filename + "\". Error:" + ex.getMessage(), ex);
		}
		return graphFromFile;
	}// end of method getGraphFromFile

	/**
	 * 
	 * @param mapName
	 */
	public void startLearningMap(final String mapName) {
		if (this.cache.get(mapName) == null) {
			throw new RuntimeException("Cant start saving map, if map is not initialized. Use getNavigationDataForMapWithChceckAndInitialization method first!");
		}
		NavigationGraphProviderForMap.startAutosavingMap(this.mapNameToFilename(mapName), this.cache.get(mapName));
	}

	/**
	 * <p>
	 * Method will try to find given graph instance in local {@link #cache} map. If found, it will use map name from
	 * {@link #cache} map and save file to filesystem. If no navgraph is not found, exception will be thrown.
	 * </p>
	 * 
	 * @param navgraph
	 * @throws IOException
	 */
	public void saveNavigationGraph(NavigationGraphSynchronized navgraph) throws IOException {
		for (Entry<String, NavigationGraphSynchronized> entry : new ArrayList<Entry<String, NavigationGraphSynchronized>>(this.cache.entrySet())) {
			if (entry.getValue() != null && entry.getValue().equals(navgraph)) {
				NavigationGraphProviderForMap.saveGraphToFile(this.mapNameToFilename(entry.getKey()), navgraph);
				return;
			}
		}// end of foreach entry in new ArrayList<Entry<String, NavigationGraphSynchronized>>(this.cache.values())
		throw new IllegalArgumentException("Given navigation graph was not found in local cache map! Use appropriete method to get nav graph!");
	}

	/**
	 * <p>
	 * Method not mean to be used. This method should be private, but can be sometimes useful. You should
	 * use {@link #startLearningMap(String)} method instead.
	 * </p>
	 * <p>
	 * Starts and tries to save selected map every one minute. Useful when learning some map and adding data to graph in
	 * any way.
	 * </p>
	 * <p>
	 * TODO no management, no stopping, or checking if more threads is saving same graph!
	 * </p>
	 * 
	 * @param filename	where to save given graph instance
	 * @param g		graph to be saved once a minute
	 */
	public static void startAutosavingMap(final String filename, final NavigationGraphSynchronized g) {
		try {
			saveGraphToFile(filename, g);
		} catch (Exception ex) {
			log.error("Error while first autosaving! I give up and end! Error:" + ex.getMessage(), ex);
			throw new RuntimeException("Error while first autosaving! I give up and end! Error:" + ex.getMessage(), ex);
		}
		Thread vlakno = new Thread(new Runnable() {
			@Override
			public void run() {
				while (true) {
					try {
						Thread.sleep(60 * 1000);
					} catch (InterruptedException ex) {
						log.info("End of saving file " + filename);
						break;
					}
					try {
						saveGraphToFile(filename, g);
					} catch (Exception ex) {
						log.error("Error while autosaving in thread! Error:" + ex.getMessage(), ex);
					}
				}
			}
		});
		vlakno.setName("Saving thread for file " + filename);
		vlakno.setDaemon(true);
		vlakno.start();
	}

	/**
	 * <p>
	 * {@link #saveNavigationGraph(NavigationGraphSynchronized)} is intend to be used mainly. This method should not be used
	 * in normal bot. It is mean to be an helpfull method for saving graphs from some utility programs.
	 * </p>
	 * 
	 * @param filename	filename of destination graphML file where to save navigation graph
	 * @param g		navigation graph which you would like to save to graphML file
	 * @throws IOException	exception is raised on error while serializing graph
	 */
	public static void saveGraphToFile(final String filename, final NavigationGraphSynchronized g) throws IOException {
		long timeStart = System.currentTimeMillis();
		CharArrayWriter aw = new CharArrayWriter(900000);
		GraphMLWriter2<NavigationGraphSynchronized, MyVertice, MyEdge> gw;
		gw = new GraphMLWriter2<NavigationGraphSynchronized, MyVertice, MyEdge>(aw, new MetadataGraphTransformer(), new MetadataNodeTransformer(), new MetadataEdgeTransformer(), null);
		long filewritingTimeStart = 0;
		try {
			log.debug("Saving graphML file.");
			gw.writeGraph(g);
			// Should avoid of incomplete files.
			filewritingTimeStart = System.currentTimeMillis();
			FileWriter fw = new FileWriter(filename);
			BufferedWriter bw = new BufferedWriter(fw);
			bw.write(aw.toCharArray());
			bw.flush();
			try {
			    bw.close();
			} catch (Exception ex) {
			    log.error("Ignorring error on close. Error: " + ex.getMessage(), ex);
			}
		} catch (GraphIOException ex) {
			log.error("GraphIOException error while saving graph to file. Error: " + ex.getMessage(), ex);
			throw new IOException("GraphIOException error while saving graph to file. Error: " + ex.getMessage(), ex);
		} catch (Exception ex) {
			log.error("General exception while sawing graph. Error:" + ex.getMessage(), ex);
			throw new IOException("General exception while sawing graph. Error:" + ex.getMessage(), ex);
		}
		long timeEnd = System.currentTimeMillis();
		log.info("Gml saved to file in " + (timeEnd - timeStart) + " ms (writing to disk was " + (timeEnd - filewritingTimeStart) + "ms).");
	}
}
