package jung.myalghoritm.dynamicWeigths;

import org.apache.commons.collections15.Transformer;

import sk.stuba.fiit.pogamut.jungigation.pathPlanners.TransformerChangeable;
import edu.uci.ics.jung.graph.Graph;

/**
 * <p>
 * Transformer for rating edge cost according cost of target vertex and edge. Just look at
 * {@link #transform(Object)} method. You can at that method count into account not only the edge
 * weight, but also target vertex weight by setting appropriate transformer.
 * </p>
 * <p>
 * Implements {@link Transformer} interface, so it can by passed directly to numerous algorithms from
 * jung library.
 * </p>
 * <p>
 * Extends {@link TransformerChangeable} class, which holds current edge transformer. Vertex
 * transformer is hold by this class internally. There are also
 * </p>
 * <p>
 * If you want to make own sum of vertex and edge price, you can do that by overriding
 * {@link #transformMerge(Number, Number)} method.
 * </p>
 * 
 * @author LuVar
 * 
 * @param <V>	vertex type
 * @param <E>	edge type
 */
public class EdgeAndVertexToNumberWeightTransformer<V, E> extends TransformerChangeable<E, Number> {
	protected Transformer<V, Number> currentTransformerInterestForVertex;
	protected final Graph<V, E> navigationGraph;

	public EdgeAndVertexToNumberWeightTransformer(Graph<V, E> navigationGraph, Transformer<V, Number> transformerVertexInterest, Transformer<E, Number> transformerEdgesInterest) {
		super(transformerEdgesInterest);
		this.navigationGraph = navigationGraph;
		this.currentTransformerInterestForVertex = transformerVertexInterest;
	}

	/**
	 * <p>
	 * Returns sum of edge price and targeted vertex price.
	 * </p>
	 * 
	 * @return	edgePrice.floatValue() + verticePrice.floatValue()
	 */
	@Override
	public final Number transform(E edge) {
		Number edgePrice = super.transform(edge);
		Number verticePrice = this.currentTransformerInterestForVertex.transform(this.navigationGraph.getDest(edge));
		return this.transformMerge(edgePrice, verticePrice);
	}

	/**
	 * <p>
	 * </p>
	 * 
	 * @param edgePrice		price for edge, received from edge transformer
	 * @param verticePrice		price for vertex, received by vertex transformer
	 * @return				summed parameters as double number
	 */
	protected Number transformMerge(Number edgePrice, Number verticePrice) {
		// TODO possible loosing precise number
		return (edgePrice.doubleValue() + verticePrice.doubleValue());
	}

	public final void setCurrentTransformerInterestForVertex(Transformer<V, Number> currentTransformerInterestForVertex) {
		this.currentTransformerInterestForVertex = currentTransformerInterestForVertex;
	}

	/**
	 * <p>
	 * Method has same effect as calling {@link #setCurrentTransformer(Transformer)} method directly.
	 * </p>
	 * 
	 * @param currentTransformerInterestForEdge
	 */
	public final void setCurrentTransformerInterestForEdge(Transformer<E, Number> currentTransformerInterestForEdge) {
		this.setCurrentTransformer(currentTransformerInterestForEdge);
	}

	public final Transformer<V, Number> getCurrentTransformerInterestForVertex() {
		return this.currentTransformerInterestForVertex;
	}

	/**
	 * 
	 * @return	returns current edge transformer, by calling {@link #getCurrentTransformer()} method
	 */
	public final Transformer<E, Number> getCurrentTransformerInterestForEdge() {
		return this.getCurrentTransformer();
	}
}
